/*
 * Copyright 2014 - 2020 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.excel.datasource;

import com.github.pjfanning.xlsx.impl.StreamingCell;
import crazydev.iccube.builder.errors.OlapBuilderErrorException;
import crazydev.iccube.builder.excel.errors.OlapExcelBuilderErrorCode;
import crazydev.iccube.builder.model.def.IOlapBuilderDataColumnDef;
import crazydev.iccube.builder.model.def.OlapBuilderAbstractTableRow;
import org.apache.poi.ss.usermodel.*;
import org.jetbrains.annotations.Nullable;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;

import java.util.Date;
import java.util.Map;

final class OlapBuilderExcelTableRow extends OlapBuilderAbstractTableRow
{
    private final OlapBuilderExcelDataTable table;

    private final Map<String, Integer> columnMapping;

    private final Row excelRow;

    private final boolean trimStrings;

    public OlapBuilderExcelTableRow(OlapBuilderExcelDataTable table, Map<String, Integer> columnMapping, Row excelRow)
    {
        this.table = table;
        this.trimStrings = table.getTrimStrings();
        this.columnMapping = columnMapping;
        this.excelRow = excelRow;
    }

    @Override
    public Object getDataImpl(IOlapBuilderDataColumnDef columnDef)
    {
        final String columnName = columnDef.getName();

        // special rownumber column
        if (OlapBuilderExcelDataTable.COL_ROWNUM_NAME.equals(columnName))
        {
            return excelRow.getRowNum() + 1;
        }

        final Integer cellPos = columnMapping.get(columnName);

        if (cellPos == null)
        {
            return null;  // it´s something that should never happen (validation should catch this error,
            // keep going and send NULL
        }

        final Cell cell = excelRow.getCell(cellPos);
        if (cell == null)
        {
            return null;
        }
        if (cell.getCellType() == CellType.FORMULA)
        {
            return evaluateAsFormula(cell, cellPos, columnDef);
        }

        final OlapBuilderExcelType cellType = OlapBuilderExcelDataTable.getCellType(cell);
        switch (cellType)
        {
            case EXCEL_NUMERIC:
                return cell.getNumericCellValue();
            case EXCEL_DATE:
                final Date javaDate = cell.getDateCellValue();
                return new LocalDateTime(javaDate.getTime());
            case EXCEL_STRING:
                final String value = cell.getStringCellValue();
                return (trimStrings && value != null) ? value.trim() : value;
            case EXCEL_BOOLEAN:
                return cell.getBooleanCellValue();
            case EXCEL_BLANK:
                return null;
            default:
                throw new OlapBuilderErrorException(OlapExcelBuilderErrorCode.EXCEL_UNSUPPORTED_CELL_TYPE,
                                                    cellType,
                                                    excelRow.getRowNum() + 1,
                                                    cellPos);
        }
    }

    @Nullable
    private Object evaluateAsFormula(Cell cell, int cellPos, IOlapBuilderDataColumnDef columnDef)
    {
        if (cell instanceof StreamingCell)
        {
            throw new OlapBuilderErrorException(OlapExcelBuilderErrorCode.EXCEL_FORMULA_EVALUATION_EXCEPTION,
                                                table.getSheetName(),
                                                cellPos,
                                                cell.getRowIndex(),
                                                "The stream reader does not support formulas in cells, you can switch off the option in the datasource");
        }

        String errorMessage;
        try
        {
            final Workbook workbook = cell.getSheet().getWorkbook();
            final FormulaEvaluator formulaEvaluator = workbook.getCreationHelper().createFormulaEvaluator();
            final CellValue cellValue = formulaEvaluator.evaluate(cell);
            final CellType cellType = cellValue.getCellType();
            switch (cellType)
            {
                case NUMERIC:
                    if (OlapBuilderExcelDataTable.isDateCell(cell))
                    {
                        final Date javaDate = DateUtil.getJavaDate(cell.getNumericCellValue());
                        return new LocalDate(javaDate.getTime());
                    }
                    else
                    {
                        return cellValue.getNumberValue();
                    }
                case BOOLEAN:
                    return cellValue.getBooleanValue();
                case STRING:
                    return cellValue.getStringValue();
                case BLANK:
                    return null;
                default:
                    errorMessage = "Unable to evaluate type [" + cellType.name() + "] -> [" + cellValue.formatAsString() + "]";
            }
        }
        catch (RuntimeException ex)
        {
            errorMessage = ex.getLocalizedMessage();
        }

        throw new OlapBuilderErrorException(OlapExcelBuilderErrorCode.EXCEL_FORMULA_EVALUATION_EXCEPTION,
                                            table.getSheetName(),
                                            cell.getRowIndex() + 1,
                                            columnDef.getName(),
                                            errorMessage);

    }

    @Override
    public boolean isEmpty()
    {
        if (excelRow == null)
        {
            return true;
        }
        for (Integer integer : columnMapping.values())
        {
            final Cell cell = excelRow.getCell(integer);
            if (cell != null && cell.getCellType() != CellType.BLANK)
            {
                return false;
            }
        }
        return true;
    }
}
