/*
 * Copyright 2014 - 2020 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.excel.datasource;

import crazydev.common.collection.CdFilter;
import crazydev.common.property.CdProperty;
import crazydev.common.property.CdReadWriteProperty;
import crazydev.iccube.builder.errors.OlapBuilderErrorException;
import crazydev.iccube.builder.model.def.IOlapBuilderConnection;
import crazydev.iccube.builder.model.def.IOlapBuilderDataSource;
import crazydev.iccube.builder.model.def.IOlapBuilderTabularDataDef;
import crazydev.iccube.builder.model.def.IOlapBuilderValidationEnabled;
import crazydev.iccube.builder.model.impl.OlapBuilderBaseDataSource;
import crazydev.iccube.builder.model.impl.table.OlapBuilderBaseDataTable;
import crazydev.iccube.builder.model.validation.OlapBuilderValidator;
import crazydev.iccube.builder.ux.meta.common.wizard.UxBuilderWizardStep;
import crazydev.iccube.builder.ux.meta.common.wizard.UxBuilderWizardValidation;
import crazydev.iccube.builder.ux.meta.datasource.UxBuilderDataSourceType;
import crazydev.iccube.builder.ux.meta.datasource.UxBuilderDataSourceTypeGroupId;
import crazydev.iccube.builder.ux.meta.datasource.wizard.UxBuilderCreateDataSourceWizardSteps;
import crazydev.iccube.builder.ux.meta.datasource.wizard.UxBuilderCreateTypedDataSourceWizard;
import crazydev.iccube.builder.ux.meta.datasource.wizard.UxBuilderDataSourceNameForm;
import crazydev.iccube.fs.OlapFileSystem;
import crazydev.iccube.olap.component.context.OlapRuntimeContext;
import crazydev.iccube.olap.loggers.OlapLoggers;
import jakarta.xml.bind.annotation.XmlAttribute;
import jakarta.xml.bind.annotation.XmlElement;
import jakarta.xml.bind.annotation.XmlRootElement;
import org.jetbrains.annotations.Nullable;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

/**
 * Datasource implementation for the <b>Excel</b> files.
 */
@XmlRootElement(name = "excelDS")
public class OlapBuilderExcelDataSource extends OlapBuilderBaseDataSource<OlapBuilderExcelConnection>
{
    public static final CdProperty FILE = new CdReadWriteProperty(OlapBuilderExcelDataSource.class, "excelFile")
    {
        @Override
        public Class<?> getTypeForDefaultEditor()
        {
            return File.class;
        }
    };

    public static final CdProperty WITHOUT_FORMULAS = new CdReadWriteProperty(OlapBuilderExcelDataSource.class, "excelWithoutFormulas");

    @XmlElement(name = "excelFileName", required = true)
    private String excelFile;

    @XmlAttribute(name = "excelWithoutFormulas", required = false)
    private boolean excelWithoutFormulas;

    public OlapBuilderExcelDataSource()
    {
        this.excelWithoutFormulas = true;
    }

    public OlapBuilderExcelDataSource(String excelFileName)
    {
        this(null, excelFileName);
    }

    public OlapBuilderExcelDataSource(@Nullable File xmlFile, String excelFile)
    {
        super(excelFile);

        this.excelFile = excelFile;

        setXmlFile(xmlFile);
    }

    @Override
    public UxBuilderDataSourceTypeGroupId getUxGroupId()
    {
        return UxBuilderDataSourceTypeGroupId.dsTypeGroupFiles;
    }

    @Override
    protected String getReportDataSourceType()
    {
        return "excel";
    }

    @Override
    protected String getReportDataSourceTypeCaption()
    {
        return "Excel";
    }

    @Override
    public boolean isDiscoverTablesSupported()
    {
        return true;
    }

    /**
     * @return a possibly null list of table that can be created by this datasource.
     */
    @Override
    @Nullable
    public List<Class<? extends IOlapBuilderTabularDataDef>> getCreatedTableTypes()
    {
        return null /* cannot create any table */;
    }

    @Override
    public OlapBuilderExcelConnection createConnection(OlapRuntimeContext context, boolean forceRefresh)
    {
        final OlapFileSystem fs = context.getRootFileSystem();
        final String d = fs.evalPropertyForCubesLocation(excelFile);
        return new OlapBuilderExcelConnection(this, d, excelWithoutFormulas);
    }

    @Override
    public OlapBuilderBaseDataTable<OlapBuilderExcelConnection> createEmptyDiscoverTable(String tableName)
    {
        return new OlapBuilderExcelDataTable(tableName);
    }

    public String getExcelFile()
    {
        return excelFile;
    }

    @Override
    public OlapBuilderValidator<IOlapBuilderValidationEnabled, IOlapBuilderDataSource> getValidator()
    {
        return new OlapBuilderExcelDataSourceValidator();
    }

    @Override
    public List<String> discoverAllTablesNames(OlapBuilderExcelConnection openedConnection, boolean filterSystemSchemas, final @Nullable String filter)
    {
        CdFilter<String> cdFilter = sheetName -> acceptTableNameForDiscoverTableNames(filter, sheetName);
        List<String> sheetNames = openedConnection.getSheetNames(cdFilter);
        return sheetNames;
    }

    /**
     * Support for caching table content when used several times. This makes no
     * sense for several datasources (e.g., in-memory, files, offline, etc...).
     */
    @Override
    public boolean acceptCachedTable()
    {
        return false /* guess not much sense for us to cache anything */;
    }

    public boolean isWithoutFormulas()
    {
        return excelWithoutFormulas;
    }

    protected UxBuilderDataSourceType createUxType()
    {
        final String id = getTypeId();

        return new UxBuilderDataSourceType(id, getUxTypeImage(), () -> new UxBuilderCreateTypedDataSourceWizard(id, createUxCreateWizardSteps())
        {
            /**
             * Validate the wizard models until the 'activeStep' possibly returning new steps.
             *
             * @throws OlapBuilderErrorException properly handled in the UI
             */
            @Override
            public UxBuilderWizardValidation validate(OlapRuntimeContext context, int activeStep)
            {
                final UxBuilderDataSourceNameForm name = (UxBuilderDataSourceNameForm) steps.get(0).getModel();
                final OlapBuilderExcelDataSource props = (OlapBuilderExcelDataSource) steps.get(1).getModel();

                props.setNameAndDescription(name.getName(), name.getDescription());

                // Ensure we can open a connection (i.e., access the file).

                IOlapBuilderConnection conn = null;

                try
                {
                    conn = props.createConnection(context, true);
                    conn.openForCheckConnection(context);

                    // Dunno: GWT was possibly doing that stuff ?
                    // dataSource.refreshCachedEntities();

                    return new UxBuilderWizardValidation(activeStep);

                }
                catch (OlapBuilderErrorException ex)
                {
                    OlapLoggers.UX_REQUEST_BUILDER.error("[ux.builder] datasource validation error", ex);

                    throw ex /* properly handled in the UI */;
                }
                finally
                {
                    if (conn != null)
                    {
                        conn.close();
                    }
                }
            }
        });
    }

    protected List<UxBuilderWizardStep> createUxCreateWizardSteps()
    {
        final List<UxBuilderWizardStep> steps = new ArrayList<>();

        steps.add(UxBuilderCreateDataSourceWizardSteps.name());
        steps.add(UxBuilderCreateDataSourceWizardSteps.propsExclude(true, createEmptyDataSource(), NAME, DESCRIPTION));

        return steps;
    }

}
