/*
 * Copyright 2014 - 2020 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.excel.datasource;

import com.github.pjfanning.xlsx.StreamingReader;
import crazydev.common.collection.CdFilter;
import crazydev.common.exception.programming.CdShouldNotBeHereProgrammingException;
import crazydev.iccube.builder.datasource.OlapBuilderAbstractConnection;
import crazydev.iccube.builder.errors.OlapBuilderErrorException;
import crazydev.iccube.builder.excel.errors.OlapExcelBuilderErrorCode;
import crazydev.iccube.fs.OlapFile;
import crazydev.iccube.fs.OlapFileSystem;
import crazydev.iccube.olap.component.context.OlapRuntimeContext;
import crazydev.iccube.olap.loggers.OlapLoggers;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.jetbrains.annotations.Nullable;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

public class OlapBuilderExcelConnection extends OlapBuilderAbstractConnection<OlapBuilderExcelDataSource>
{
    private final String untrustedExcelFile;

    private final boolean dontUseStream;

    protected Workbook workbook;

    public OlapBuilderExcelConnection(OlapBuilderExcelDataSource dataSource, String untrustedExcelFile, boolean withoutFormulas)
    {
        super(dataSource);

        this.untrustedExcelFile = untrustedExcelFile;
        this.dontUseStream = !withoutFormulas;
    }

    public static OlapBuilderExcelConnection openFromInputStream(InputStream content, String fileName, boolean withoutFormulas)
    {
        final OlapBuilderExcelConnection connection = new OlapBuilderExcelConnection(new OlapBuilderExcelDataSource(), fileName, withoutFormulas);

        connection.createWorkbook(() -> content, null, connection.untrustedExcelFile);
        connection.forceIsOpen();

        return connection;
    }

    @Override
    public void onOpen(OlapRuntimeContext context) throws OlapBuilderErrorException
    {
        final OlapFileSystem fs = context.getRootFileSystem();

        // For security reason we must create a file within the constraint file system.

        final OlapFile trusted = fs.create(untrustedExcelFile);

        OlapLoggers.BUILDER.info("[excel] opening [stream:" + !dontUseStream + "] [" + untrustedExcelFile + "]");

        createWorkbook(trusted::createInputStream, trusted.__getUnderlying(), trusted.getUnderlyingAbsolutePathForUserError());

    }

    @Override
    public boolean isInvalid()
    {
        return !dontUseStream /* a stream => cannot re-use it */;
    }

    private void createWorkbook(MySupplier<InputStream> inputStream, @Nullable File file, String fileNameForUser)
    {
        try
        {
            if (!dontUseStream)
            {
                try
                {
                    workbook = StreamingReader.builder()
                            .rowCacheSize(100)    // number of rows to keep in memory (defaults to 10)
                            .bufferSize(4096)     // buffer size to use when reading InputStream to file (defaults to 1024)
                            .open(inputStream.get());  // InputStream or File for XLSX file (required)
                    return;
                }
                catch (Exception ignore)
                {

                }
            }

            if (file != null)
            {
                workbook = WorkbookFactory.create(file);
            }
            else
            {
                //more memory consuming but not choice if we've only a stream (e.g Google http)
                workbook = WorkbookFactory.create(inputStream.get());
            }
        }
        catch (FileNotFoundException ex)
        {
            throw new OlapBuilderErrorException(ex, OlapExcelBuilderErrorCode.EXCEL_FILE_NOT_FOUND, fileNameForUser, ex.getMessage());
        }
        catch (IOException ex)
        {
            throw new OlapBuilderErrorException(ex, OlapExcelBuilderErrorCode.EXCEL_FILE_READING_ERROR, fileNameForUser, ex.getMessage());
        }
        catch (IllegalArgumentException ex)
        {
            throw new OlapBuilderErrorException(ex, OlapExcelBuilderErrorCode.EXCEL_FILE_INCORRECT_CONTENT, fileNameForUser, ex.getMessage());
        }
    }

    @Override
    protected void onClose()
    {
        // free memory
        if (workbook != null)
        {
            try
            {
                workbook.close();
            }
            catch (IOException ignore)
            {

            }
        }
        workbook = null;
    }

    public Workbook _getWorkbook()
    {
        assertIsOpen();
        return workbook;
    }

    private Sheet getSheet(String sheetName)
    {
        assertIsOpen();
        final Sheet sheet = workbook.getSheet(sheetName);
        if (sheet == null)
        {
            throw new OlapBuilderErrorException(OlapExcelBuilderErrorCode.EXCEL_SHEET_NOT_FOUND, sheetName, untrustedExcelFile);
        }
        return sheet;
    }

    @Nullable
    public Sheet getSheet(OlapBuilderExcelDataTable dataTable)
    {
        return getSheet(dataTable.getSheetName());
    }

    public List<String> getSheetNames(CdFilter<String> filter)
    {
        final List<String> dataTables = new ArrayList<String>();
        for (Sheet sheet : workbook)
        {
            final String sheetName = sheet.getSheetName();
            if (filter.accept(sheetName))
            {
                dataTables.add(sheetName);
            }
        }
        return dataTables;
    }

    public Accessor getSheetAccessor(String sheetName, final int headerRowIdx)
    {
        assertIsOpen();
        return new Accessor(getSheet(sheetName), headerRowIdx);
    }

    public void checkSheetExists(String sheetName)
    {
        assertIsOpen();
        if (getSheetNames(sheetName::equals).isEmpty())
        {
            throw new OlapBuilderErrorException(OlapExcelBuilderErrorCode.EXCEL_SHEET_NOT_FOUND, sheetName);
        }
    }

    @FunctionalInterface
    public interface MySupplier<T>
    {
        T get() throws IOException;
    }

    static class Accessor
    {
        private final Iterator<Row> iter;

        private final Row headerRow;

        private final Row firstDataRow;

        Accessor(Sheet sheet, Integer headerRowIdx)
        {
            iter = sheet.iterator();
            headerRow = getHeaderRow(headerRowIdx);
            firstDataRow = iter.hasNext() ? iter.next() : null;
        }

        boolean isEmptySheet()
        {
            return headerRow == null;
        }

        private Row getHeaderRow(Integer headerRowIdx)
        {
            int count = 0;
            while (iter.hasNext())
            {
                Row row = iter.next();
                if (row == null)
                {
                    throw new CdShouldNotBeHereProgrammingException();
                }
                if (headerRowIdx == count++)
                {
                    return row;
                }
            }
            return null;
        }

        public Row getHeaderRow()
        {
            return headerRow;
        }

        public Row getFirstDataRow()
        {
            return firstDataRow;
        }

        Iterator<Row> iterator()
        {
            return new Iterator<Row>()
            {
                boolean gotFirst = false;

                @Override
                public boolean hasNext()
                {
                    return gotFirst ? iter.hasNext() : firstDataRow != null;
                }

                @Override
                public Row next()
                {
                    if (gotFirst)
                    {
                        return iter.next();
                    }
                    gotFirst = true;
                    return firstDataRow;
                }
            };

        }

    }
}
