## Java Process Memory Usage

The icCube server is implemented in Java. This section is describing the runtime of a Java process regarding
its memory usage and eventually giving some advice to **determine the maximum amount of memory required**
by a running icCube server. If you're familiar enough with Java, you can skip directly to
this [section](#heap--how-to-determine-the-maximum-value---xmx--) at the end of this document.

This document lists the following sections:
  
  - [Garbage Collected Runtime](#garbage-collected-runtime)
  - [JVM Native Memory](#jvm-native-memory)
  - [PID, RAM \& SWAP](#pid-ram--swap)
  - [Heap : Maximum, Total, Free](#heap--maximum-total-free)
  - [Java Out Of Memory](#java-out-of-memory)
  - [Process Out Of Memory](#process-out-of-memory)
  - [Swap](#swap)
  - [Heap : How to Determine the Maximum Value ( -Xmx ) ?](#heap--how-to-determine-the-maximum-value---xmx--)

Please refer to this [page](../MemoryUsage.md) for a series of documents related to the memory usage in icCube.

### Garbage Collected Runtime

Java is a garbage-collected language meaning the Java runtime (aka. JVM) is managing memory automatically
by tracking and reclaiming memory that is no longer in use by the program. The process of identifying and
releasing memory that is no longer needed is called `garbage collection`.

This managed memory is allocated from the Java `heap` which is a specific area of the process used for dynamic
memory allocation. This heap is managed by the `garbage collector`. The maximum amount of memory used for the
heap can be defaulted by the JVM at startup time but most of the time it is controlled by the `-Xmx` parameter.

For example, the following is starting a Java process with a maximum of `1g` :

```
  # java -Xmx1g ....
```

**-Xmx defines the heap memory used by the Java process not the total memory used by the Java process**.

Besides the heap there are other pools of memory managed by the JVM :

- `Permanent Generation` : the memory holding objects such as classes and methods (as specified by the`-XX:MaxPermSize`
  parameter).

- `Thread Stack` : the memory allocated for the stack of each thread (as specified by the `-Xss` parameter).

**Roughly, the managed memory usage of an icCube server can be predicted with the formula :**

```
  = [-Xmx] + [-XX:MaxPermSize] + number_of_threads * [-Xss]
```

**Besides the memory consumed by the Java application the JVM itself requires some internal (aka. native)
memory for its own behavior.** This explains that a Java program started with `-Xm512m` can consume much more than
`512m` of RAM.

The following is showing the Java process consuming `866m` (865984 bytes) :

```
  # top -p 103247
   
     PID USER      PR  NI    VIRT    RES    SHR S  %CPU  %MEM     TIME+ COMMAND
  103247 guest     20   0 8611300 865984  33912 S   0.3   1.3   0:21.34 java
```

The next section is giving a brief overview of those internal memories.

### JVM Native Memory

Monitoring the actual memory used by a process depends on the OS being used. On Linux, using `top` is for example
a quick way to check `RES` memory and see that this value can be much larger than the `-Xmx` parameter value.
The [Native Memory Tracking](https://docs.oracle.com/javase/8/docs/technotes/guides/troubleshoot/tooldescr007.html)
utility allows for understanding where this 'extra' memory is coming from. This tracking must be activated when
starting the JVM :

```
  # java -Xmx512m -XX:NativeMemoryTracking=summary ...
```    

Note that this tracking is adding some performance overhead so use it only to troubleshoot your installation.

Then you can use the `jcmd PID ...` command to trigger a dump :

```
  # jcmd 125410  VM.native_memory summary scale=MB

  125410:
  
  Native Memory Tracking:
  
  (Omitting categories weighting less than 1MB)
  
  Total: reserved=2092MB, committed=745MB
  -                 Java Heap (reserved=512MB, committed=512MB)
                              (mmap: reserved=512MB, committed=512MB) 
  
  -                     Class (reserved=1025MB, committed=11MB)
                              (classes #14777)
                              (  instance classes #13736, array classes #1041)
                              (malloc=1MB #29826) 
                              (mmap: reserved=1024MB, committed=9MB) 
                              (  Metadata:   )
                              (    reserved=64MB, committed=60MB)
                              (    used=59MB)
                              (    waste=1MB =1.32%)
                              (  Class space:)
                              (    reserved=1024MB, committed=9MB)
                              (    used=9MB)
                              (    waste=1MB =5.58%)
  
  -                    Thread (reserved=115MB, committed=9MB)
                              (thread #116)
                              (stack: reserved=115MB, committed=9MB)
  
  -                      Code (reserved=243MB, committed=25MB)
                              (malloc=2MB #9020) 
                              (mmap: reserved=242MB, committed=24MB) 
  
  -                        GC (reserved=80MB, committed=80MB)
                              (malloc=29MB #9330) 
                              (mmap: reserved=51MB, committed=51MB) 
  
  -                  Internal (reserved=1MB, committed=1MB)
                              (malloc=1MB #11749) 
  
  -                     Other (reserved=1MB, committed=1MB)
                              (malloc=1MB #35) 
  
  -                    Symbol (reserved=15MB, committed=15MB)
                              (malloc=13MB #378485) 
                              (arena=2MB #1)
  
  -    Native Memory Tracking (reserved=7MB, committed=7MB)
       (tracking overhead=7MB)
  
  -        Shared class space (reserved=16MB, committed=12MB)
                              (mmap: reserved=16MB, committed=12MB) 
  
  -               Arena Chunk (reserved=12MB, committed=12MB)
                              (malloc=12MB) 
  
  -                 Metaspace (reserved=64MB, committed=60MB)
                              (mmap: reserved=64MB, committed=60MB)
```

From the output : `Total: reserved=2092MB, committed=745MB`.

This means `745m` are committed, and the JVM might use up to `2092m`. The committed value will depend on the actual
usage of icCube (e.g., more threads, more complex internal allocated objects, etc...). More on this later in this 
document.

**Overhead**

This native memory overhead is quite high for a small heap but note that: the bigger the Java heap, the smaller 
the overhead in percentage. For example, you can have `1-2g` overhead for a `32g` heap.

### PID, RAM & SWAP

As a reminder, the `Server Status` is showing the `PID` of the icCube process and both physical memory and swap
available on the machine :

![OS](images/os-status.png)

### Heap : Maximum, Total, Free

The Java heap status can be represented by three values at any point of time : `Maximum`, `Total`, and `Free` memory.

- `Maximum` : this value is the one as specified by the `-Xmx` parameter. The JVM is not going to use more than this
  value for the heap.
- `Total` : the actual size of the heap (it cannot be greater than `Maximum`).
- `Free` : part of the allocated `Total` being not used by the Java program.

The JVM is free to use up to `Maximum` memory for its heap; the JVM is going to allocate the memory as needed up to
`Maximum` value. This is why the `Total` might not be equal to the `Maximum` value. It's left up to the JVM to give
back to the OS some previously allocated memory (not mandatory).

The icCube `Used Memory` as shown in the Admin page is equal to : `Total - Free` :

![Used Memory](images/admin-used-memory.png)

The `Server Status` is showing the current `Maximum`, `Total`, and `Free` memory :

![Memory Status](images/server-status.png)

You can as well get an overview of the historical values using `grep -GC icCube*log` :

```
... (12-09-23 11:34:43.243 UTC) [GC] (G1 Young Generation) : 40ms ( free:717MB / total:1736MB / max:3072MB )
... (12-09-23 11:47:42.837 UTC) [GC] (G1 Young Generation) : 42ms ( free:726MB / total:1736MB / max:3072MB )
... (12-09-23 11:48:00.461 UTC) [GC] (G1 Young Generation) : 25ms ( free:728MB / total:1736MB / max:3072MB )
... (12-09-23 12:19:16.977 UTC) [GC] (G1 Young Generation) : 86ms ( free:712MB / total:1736MB / max:3072MB )
... (12-09-23 12:40:58.467 UTC) [GC] (G1 Young Generation) : 69ms ( free:657MB / total:1736MB / max:3072MB )
```

Note that the REST [API](../../server/api/ServerStatus.md) allows for retrieving those values.

Please refer to this [page](MemoryJavaProcessGC.md) for additional considerations regarding the
Java garbage collector.

### Java Out Of Memory

A Java `Out Of Memory` is triggered when the JVM cannot allocate anymore some memory from its heap : the heap is full.
The process itself will **not crash** assuming there is enough RAM (and possibly swap) on the machine for both the heap
and the native memory. The Admin interface will display a red bell (top right corner) asking to restart the server.

![OOM](images/oom.png)

Indeed, there is no guarantee about the consistency of the schema data if this error occurred when a schema was being
loaded. Note that if this error happened while processing an MDX statement, then the server should be safe and there
is no need to restart immediately.

### Process Out Of Memory

A process `Out Of Memory` is triggered when the OS cannot allocate anymore RAM (heap and/or native memory) and the
process is going to **crash**. To avoid that, you must ensure that the total amount of memory (i.e., heap + native
memory) does not exceed the amount of RAM (and swap) available in the system (assuming icCube is the main process in
the system).

This `Out Of Memory` can be triggered even if the Java heap is partially filled. Indeed, this partial heap plus the
native memory may already exceed the available RAM.

On Linux, you can troubleshoot such a crash by analyzing the content of `syslog` as following :

```
  # more /var/log/syslog | grep -i oom       
  # more /var/log/syslog | grep -i kill 
```

### Swap

As a reminder, in low memory servers (e.g., cloud servers) adding some swap is handy to avoid a process
`Out Of Memory` and therefore a crash. If you find that icCube's performance is suboptimal, you can then monitor 
the usage of this swap space.

Note that even when there's plenty of RAM, having some swap space comes in handy to prevent the RAM from
getting cluttered with potential memory leaks.

### Heap : How to Determine the Maximum Value ( -Xmx ) ?

There is no magic formula giving the exact value of the `-Xmx` parameter. The Java **heap must
be large enough** mainly :

- to hold all the loaded [schemas](MemorySchema.md),
- to be able to reload schemas (keeping previously loaded schemas) if needed,
- to be able to process MDX queries.

If the heap is not large enough then icCube is going to experience a Java `Out Of Memory` [error](#java-out-of-memory).

At the same time, **a full [heap](#garbage-collected-runtime) plus the [native memory](#jvm-native-memory) must not
exceed the amount of RAM (and swap) available** in the system. Otherwise, icCube is going to experience a process
`Out Of Memory` error and therefore a crash.

We're advising to set up a **staging environment** that is reproducing as much as possible the production environment
to learn/monitor how much memory (heap + native memory) is required by icCube. You can use this environment for stress
testing the system and check how icCube behaves in your environment.

At the same time **monitor the actual size of the icCube process itself** (e.g., using `ps`, `top`, etc...) to 
ensure the heap + native memory is not exceeding the machine RAM (and swap). If you're curious, you can investigate
this native memory using the [Native Memory Tracking](#jvm-native-memory) utility.

### See Also

Please refer to this [page](../MemoryUsage.md) for a series of documents related to the memory usage in icCube.

### Contact Us

Do not hesitate to [contact us](https://www.iccube.com/contact-us/) for some help configuring the memory.

_