/*
 * Copyright 2014 - 2018 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.mongodb.datatable.distinct;

import com.mongodb.BasicDBObject;
import com.mongodb.client.DistinctIterable;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoCursor;
import com.mongodb.client.MongoDatabase;
import crazydev.iccube.builder.OlapBuilderConnectionPool;
import crazydev.iccube.builder.OlapBuilderContext;
import crazydev.iccube.builder.datasource.reader.OlapBuilderAbstractTableRowReader;
import crazydev.iccube.builder.model.def.IOlapBuilderDataColumnDef;
import crazydev.iccube.builder.model.def.IOlapBuilderTableRow;
import crazydev.iccube.builder.mongodb.common.OlapBuilderMongoDbHelper;
import crazydev.iccube.builder.mongodb.datasource.OlapBuilderMongoDbConnection;
import crazydev.iccube.builder.mongodb.datatable.common.OlapBuilderMongoDbTableRowReader;
import org.bson.BsonValue;
import org.bson.Document;
import org.jetbrains.annotations.Nullable;

class OlapBuilderMongoDbDistinctTableRowReader extends OlapBuilderAbstractTableRowReader<OlapBuilderMongoDbDistinctDataTable, OlapBuilderMongoDbConnection>
{
    private MongoCursor<BsonValue> iterator;

    OlapBuilderMongoDbDistinctTableRowReader(OlapBuilderContext context,
                                             OlapBuilderConnectionPool connectionPool,
                                             int maxRowCount,
                                             OlapBuilderMongoDbDistinctDataTable table,
                                             String fullNameForEndUser)
    {
        super(context, connectionPool, maxRowCount, table, fullNameForEndUser);
    }

    static BsonValue firstNotNull(OlapBuilderMongoDbConnection connection, OlapBuilderMongoDbDistinctDataTable table)
    {
        final MongoDatabase bd = connection.getMongoDataBase();
        final MongoCollection<Document> collection = bd.getCollection(table.getCollection());

        // Query
        DistinctIterable<BsonValue> distinctQuery = buildQuery(table, collection);
        // projection
        final MongoCursor<BsonValue> iter = distinctQuery.iterator();
        while (iter.hasNext())
        {
            final BsonValue val = iter.next();
            if (val != null)
            {
                iter.close();
                return val;
            }
        }
        return null;
    }

    private static DistinctIterable<BsonValue> buildQuery(OlapBuilderMongoDbDistinctDataTable table, MongoCollection<Document> collection)
    {
        DistinctIterable<BsonValue> query = collection.distinct(table.getFieldName(), toBson("query", table.getQuery()), BsonValue.class);
        if (table.getBatchSize() != null)
        {
            query = query.batchSize(table.getBatchSize());
        }
        return query;
    }

    private static BasicDBObject toBson(String fieldName, String bson)
    {
        return OlapBuilderMongoDbTableRowReader.toBson(fieldName, bson);
    }

    @Override
    public boolean isRowSafe()
    {
        return true;
    }

    @Override
    public void doInit()
    {
        initConnection();

        final MongoDatabase bd = connection.getMongoDataBase();
        final MongoCollection<Document> collection = bd.getCollection(table.getCollection());

        // Query
        final DistinctIterable<BsonValue> distinctQuery = buildQuery(table, collection);

        iterator = distinctQuery.iterator();
    }

    @Override
    public @Nullable IOlapBuilderTableRow doNextRow()
    {
        if (!iterator.hasNext())
        {
            return null;
        }

        final BsonValue bsonValue = iterator.next();

        return new IOlapBuilderTableRow()
        {
            final BsonValue value = bsonValue;

            @Override
            public @Nullable Comparable getJavaNativeDataValue(IOlapBuilderDataColumnDef columnDef)
            {
                return OlapBuilderMongoDbHelper.toValue(columnDef.getType(), value);
            }
        };
    }

}
