/*
 * Copyright 1999 - 2014 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.mongodb.datatable.common;

import com.mongodb.BasicDBObject;
import com.mongodb.DBObject;
import crazydev.iccube.builder.model.def.IOlapBuilderDataColumnDef;
import crazydev.iccube.builder.model.def.IOlapBuilderDataTableDef;
import crazydev.iccube.builder.model.def.OlapBuilderAbstractTableRow;
import crazydev.iccube.builder.type.OlapBuilderColumnDecorationType;
import crazydev.iccube.builder.type.OlapBuilderInputType;
import org.bson.types.BasicBSONList;
import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.base.BaseLocal;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

public class OlapBuilderMongoDbTableRow extends OlapBuilderAbstractTableRow
{
    private final static DateTimeZone UTC = DateTimeZone.UTC;

    private final OlapBuilderMongoDbTableRowReaderOld rowReader;

    private final Object[] values;

    private OlapBuilderMongoDbTableRow(OlapBuilderMongoDbTableRowReaderOld rowReader, Object[] values)
    {
        this.rowReader = rowReader;
        this.values = values;
    }

    public static OlapBuilderMongoDbTableRow create(OlapBuilderMongoDbTableRowReaderOld rowReader, DBObject document)
    {
        final IOlapBuilderDataTableDef table = rowReader.getTable();
        final List<IOlapBuilderDataColumnDef> columns = table.getAllColumns();

        final Object[] values = new Object[columns.size()];

        for (IOlapBuilderDataColumnDef column : columns)
        {
            if (column.isSelected())
            {
                final int index = column.getIndex();

                values[index] = getValue(rowReader, document, column);
            }
        }

        return new OlapBuilderMongoDbTableRow(rowReader, values) /* can be used in a _ separate _ thread */;
    }

    @Nullable
    @Override
    public Object getDataImpl(IOlapBuilderDataColumnDef columnDef)
    {
        final int index = columnDef.getIndex();

        if (index >= 0 && index < values.length)
        {
            return values[index];
        }

        throw new RuntimeException("internal error: invalid index [" + index + "] [table:" + rowReader.getTableName() + "] [col:" + columnDef.getName() + "] [max:" + values.length + "]");
    }

    @Nullable
    private static Object getValue(OlapBuilderMongoDbTableRowReaderOld rowReader, DBObject document, IOlapBuilderDataColumnDef columnDef)
    {
        final String[] names = rowReader.getColumnSplit(columnDef);

        Object value = fromObject(columnDef, names, 1, document.get(names[0]));

        if (value != null)
        {
            if (columnDef.getType().isString() && columnDef.getDecorationType() != OlapBuilderColumnDecorationType.ARRAY)
            {
                value = value.toString();
            }
        }

        if (value instanceof Date)
        {
            final DateTimeZone dateTimeTZ = rowReader.getDateTimeTZ();
            final boolean applyDateTimeTZtoDate = rowReader.applyDateTimeTZtoDate();

            if (columnDef.getType().equals(OlapBuilderInputType.DATETIME))
            {
                value = applyTimeZoneProcessing(true, (Date) value, dateTimeTZ);
            }
            else if (columnDef.getType().equals(OlapBuilderInputType.DATE))
            {
                if (applyDateTimeTZtoDate)
                {
                    value = applyTimeZoneProcessing(false, (Date) value, dateTimeTZ);
                }
                else
                {
                    value = applyTimeZoneProcessing(false, (Date) value, UTC);
                }
            }
        }
        return value;
    }

    private static Object applyTimeZoneProcessing(boolean dateTime, Date value, @Nullable DateTimeZone dateTimeTZ)
    {
        // The converters are doing the following :
        //
        //   new LocalDateTime( value );
        //   new LocalDate( value );
        //
        // using the default TZ of the icCube server.

        if (dateTimeTZ == null /* server default TZ */)
        {
            return value;
        }

        // BSON : UTC datetime ( the int64 is UTC milliseconds since the Unix epoch )

        final long epoch = value.getTime();

        final DateTime dt = new DateTime(epoch, dateTimeTZ);

        final BaseLocal res;

        if (dateTime)
        {
            res = dt.toLocalDateTime();
        }
        else
        {
            res = dt.toLocalDate();
        }

        return res;
    }

    @Nullable
    private static Object fromObject(IOlapBuilderDataColumnDef columnDef, String[] names, int startPos, Object rootValue)
    {
        Object value = rootValue;
        for (int i = startPos; i < names.length; i++)
        {
            if (value instanceof BasicDBObject)
            {
                value = ((BasicDBObject) value).get(names[i]);
            }
            else if (value instanceof BasicBSONList)
            {
                // we break as we need a special handling for arrays
                value = fromList(columnDef, (BasicBSONList) value, names, i);
                break;
            }
            else
            {
                value = null;
                break;
            }
        }
        return value;
    }

    @Nullable
    private static Object fromList(IOlapBuilderDataColumnDef columnDef, BasicBSONList list, String[] names, int nextPos)
    {
        ArrayList<Object> outList = new ArrayList<>(list.size()); // ok, unlikely nested lists..
        for (int i = 0; i < list.size(); i++)
        {
            final Object dbItem = list.get(i);
            final Object object = fromObject(columnDef, names, nextPos, dbItem);
            if (object instanceof ArrayList)
            {
                // remove support for arrays of arrays ?
                outList.addAll((ArrayList) object);
            }
            else
            {
                if (object == null)
                {
                    if (nextPos >= names.length - 1)  // only for terminal
                    {
                        final Comparable nullObject = columnDef.getNullObject();
                        if (nullObject != null)
                        {
                            outList.add(nullObject);
                        }
                    }
                }
                else
                {
                    outList.add(object);
                }
            }
        }

        return outList;
    }

}
