/*
 * Copyright 2014 - 2019 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.mongodb.datatable.common;

import com.mongodb.Block;
import com.mongodb.client.DistinctIterable;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoDatabase;
import crazydev.common.exception.programming.CdShouldNotBeHereProgrammingException;
import crazydev.common.utils.CdDefaultEnumerableForUi;
import crazydev.common.utils.CdEnumerableForUi;
import crazydev.common.utils.CdStringUtils;
import crazydev.common.xml.CdLocaleXmlAdapter;
import crazydev.iccube.builder.OlapBuilderConnectionPool;
import crazydev.iccube.builder.OlapBuilderContext;
import crazydev.iccube.builder.datasource.common.OlapBuilderTableRequestedPartitionKeys;
import crazydev.iccube.builder.datasource.reader.IOlapBuilderTablePartitionKey;
import crazydev.iccube.builder.datasource.reader.OlapBuilderAbstractTablePartitionKey;
import crazydev.iccube.builder.errors.OlapBuilderErrorCode;
import crazydev.iccube.builder.errors.OlapBuilderErrorException;
import crazydev.iccube.builder.errors.OlapBuilderErrorManager;
import crazydev.iccube.builder.model.def.IOlapBuilderDataColumnDef;
import crazydev.iccube.builder.model.impl.table.OlapBuilderBaseDataTable;
import crazydev.iccube.builder.mongodb.common.OlapBuilderMongoDbHelper;
import crazydev.iccube.builder.mongodb.datasource.OlapBuilderMongoDbConnection;
import crazydev.iccube.builder.mongodb.datasource.OlapBuilderMongoDbDataSource;
import crazydev.iccube.builder.type.OlapBuilderInputType;
import crazydev.iccube.olap.component.context.OlapRuntimeContext;
import jakarta.xml.bind.annotation.XmlElement;
import jakarta.xml.bind.annotation.adapters.XmlJavaTypeAdapter;
import org.bson.BsonValue;
import org.bson.Document;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.regex.Matcher;

public abstract class OlapBuilderMongoDbDataTable extends OlapBuilderBaseDataTable<OlapBuilderMongoDbConnection>
{
    /**
     * Kept for reading old schema (otherwise JAXB error).
     */
    @XmlJavaTypeAdapter(value = CdLocaleXmlAdapter.class)
    @XmlElement(name = "stringConverterLocale", required = false)
    private Locale stringConverterLocale;

    /**
     * Kept for reading old schema (otherwise JAXB error).
     */
    @XmlElement(required = false)
    private String stringDateConverter;

    public OlapBuilderMongoDbDataTable()
    {
    }

    public OlapBuilderMongoDbDataTable(String tableName)
    {
        super(tableName);
    }

    @Override
    public boolean isNameReadOnly()
    {
        return false;
    }

    /**
     * Adapting our collection to a UI enumerable : keep it simple for now assuming not many collections.
     */
    protected CdEnumerableForUi getCollectionAsEnumerableForUi(String collection)
    {
        final OlapBuilderMongoDbDataSource dataSource = (OlapBuilderMongoDbDataSource) getDataSource();

        if (dataSource == null)
        {
            return new CdDefaultEnumerableForUi(collection, Arrays.asList());
        }

        final List<String> names = dataSource.getCollectionNamesForUi();
        return new CdDefaultEnumerableForUi(collection, names);
    }

    @Override
    public List<? extends IOlapBuilderDataColumnDef> doDiscoverAllColumnsForRefresh(OlapRuntimeContext context, OlapBuilderMongoDbConnection connection, OlapBuilderErrorManager errorManager)
    {
        return discoverAllColumns(context, connection, errorManager);
    }

    protected abstract Object findOneForDiscoverColumns(OlapBuilderMongoDbConnection connection);

    public boolean getCacheStringValue()
    {
        final String options = getOptions();
        return options != null && options.toUpperCase().contains("CACHESTRINGVALUE");
    }

    @Nullable
    public String getOptions()
    {
        return null;
    }

    @Override
    protected List<IOlapBuilderTablePartitionKey> doGetPartitionKeys(OlapBuilderContext context, OlapBuilderConnectionPool connectionPool, @Nullable OlapBuilderTableRequestedPartitionKeys requestedPartitions)
    {
        final OlapBuilderMongoDbConnection connection = connectionPool.getOrCreateOpenedConnection(context.asRuntimeContext(), getDataSource());
        final MongoDatabase bd = connection.getMongoDataBase();
        final MongoCollection<Document> collection = bd.getCollection(getCollection());

        List<IOlapBuilderTablePartitionKey> values = new ArrayList<>();

        getDistinct(collection).forEach((Block<BsonValue>) bsonValue -> {

            final OlapBuilderInputType type = OlapBuilderMongoDbHelper.toType(bsonValue);
            final Comparable value = OlapBuilderMongoDbHelper.toValue(type, bsonValue);

            final String asString = type.getTypeConverter().toStringForPartitionKey(value);

            if (requestedPartitions == null || requestedPartitions.getKeys().indexOf(asString) != -1)
            {
                if (bsonValue == null)
                {
                    throw new OlapBuilderErrorException(OlapBuilderErrorCode.PARTITION_KEY_NULL_NOT_SUPPORTED, getUnderlyingTables(), getPartitioningColumnName());
                }

                values.add(new PartitionKey(type, asString, bsonValue));
            }
        });

        return values;
    }

    protected DistinctIterable<BsonValue> getDistinct(MongoCollection<Document> collection)
    {
        throw new CdShouldNotBeHereProgrammingException();
    }

    protected String getCollection()
    {
        throw new CdShouldNotBeHereProgrammingException();
    }

    @Nullable
    protected String addMarkers(String statement, @Nullable Comparable incrLoadMarker, @Nullable Object partitionValue)
    {
        if (CdStringUtils.isNullOrBlank(statement))
        {
            return null;
        }
        //     datetime ==>    { $date : "$ic3incrValueZ" }    LocalDateTime.toString()  is  ISO8601
        //                       ^                     ^
        //                       ^                  required ( but ISO8601 .toString does not display it )
        //                       ^
        //                     JAVA driver specific to send a datetime
        //
        //     date     ==>    dunno but does it make any sense to have a incr-load on date ?

        String json = replaceAll(statement, "$ic3incrValue", incrLoadMarker);
        json = replaceAll(json, "$ic3partitionValue", partitionValue);

        return json;
    }

    private String replaceAll(String statement, String tag, @Nullable Object value)
    {
        if (value == null)
        {
            return statement;
        }
        final String regexp = Matcher.quoteReplacement(tag);
        final String valueAsString = value.toString();
        return statement.replaceAll(regexp, valueAsString);
    }

    static class PartitionKey extends OlapBuilderAbstractTablePartitionKey
    {
        private BsonValue bsonValue; // for logging purpose

        public PartitionKey(OlapBuilderInputType type, String asString, BsonValue bsonValue)
        {
            super(type, asString);
            this.bsonValue = bsonValue;
        }

    }

}
