/*
 * Copyright 1999 - 2014 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.mongodb.datatable.aggregate;

import com.mongodb.client.AggregateIterable;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoCursor;
import com.mongodb.client.MongoDatabase;
import crazydev.iccube.builder.OlapBuilderConnectionPool;
import crazydev.iccube.builder.OlapBuilderContext;
import crazydev.iccube.builder.datasource.common.OlapBuilderTableResolvedRow;
import crazydev.iccube.builder.model.def.IOlapBuilderDataColumnDef;
import crazydev.iccube.builder.model.def.IOlapBuilderTableRow;
import crazydev.iccube.builder.mongodb.datatable.common.OlapBuilderMongoDbTableRowReader;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.jetbrains.annotations.Nullable;

import java.util.List;

public class OlapBuilderMongoDbAggregateExTableRowReader extends OlapBuilderMongoDbTableRowReader<OlapBuilderMongoDbAggregateExDataTable>
{
    private MongoCursor<?> result;

    private List<IOlapBuilderDataColumnDef> columns;

    public OlapBuilderMongoDbAggregateExTableRowReader(OlapBuilderContext context,
                                                       OlapBuilderConnectionPool connectionPool,
                                                       int maxRowCount,
                                                       OlapBuilderMongoDbAggregateExDataTable table,
                                                       String fullNameForEndUser)
    {
        super(context, connectionPool, maxRowCount, table, fullNameForEndUser);
    }

    @Override
    public void doInit()
    {
        initConnection();

        final String collectionName = table.getCollection();

        final MongoDatabase db = connection.getMongoClient().getDatabase(connection.getDataSource().getDbName());
        final MongoCollection<Document> collection = db.getCollection(collectionName);

        final List<Bson> pipeline = toBsonList("aggr. pipeline", "BsonList", table.getPipeline());

        AggregateIterable<?> aggregateQuery = collection.aggregate(pipeline);

        if (table.getBatchSize() != null)
        {
            aggregateQuery = aggregateQuery.batchSize(table.getBatchSize());
        }

        if (table.getAllowDiskUse() != null)
        {
            aggregateQuery = aggregateQuery.allowDiskUse(table.getAllowDiskUse());
        }

        result = aggregateQuery.iterator();
    }

    @Override
    @Nullable
    public IOlapBuilderTableRow doNextRow()
    {
        if (!result.hasNext())
        {
            return null;
        }

        final Object next = result.next();
        final IOlapBuilderTableRow row = toBuilderRow(next);

        return row;
    }

    private IOlapBuilderTableRow toBuilderRow(Object next)
    {
        if (next instanceof Document)
        {
            return toBuilderRowFromDocument((Document) next);
        }

        throw new RuntimeException("unexpected MongoDB result [" + (next != null ? next.getClass().getName() : "<null>") + "+");
    }

    private IOlapBuilderTableRow toBuilderRowFromDocument(Document document)
    {
        // ---------------------------------------------------------------------------------------------------------
        // Assuming this is for a "small" result table (sending a huge amount in the reporting widgets
        // makes no sense. And waiting for feedback on array handling. This implementation remains simple
        // without trying to optimize intermediate lookup (=> same objects being used several time).
        // ---------------------------------------------------------------------------------------------------------

        if (columns == null)
        {
            columns = table.discoverAllColumns(document);
        }

        final Comparable[] values = OlapBuilderMongoDbAggregateExTableDecoder.decode(columns, document);

        return new OlapBuilderTableResolvedRow(values);
    }

    @Override
    public void done()
    {
        if (result != null)
        {
            try
            {
                result.close();
            }
            catch (Exception ignore)
            {
                //yes
            }
        }

        super.done();
    }

}
