/*
 * Copyright 1999 - 2014 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.mongodb.datatable.aggregate;

import crazydev.common.collection.CdComparableArray;
import crazydev.iccube.builder.errors.OlapBuilderErrorCode;
import crazydev.iccube.builder.errors.OlapBuilderErrorException;
import crazydev.iccube.builder.model.def.IOlapBuilderDataColumnDef;
import crazydev.iccube.builder.type.IOlapBuilderInputTypeConverter;
import crazydev.iccube.builder.type.OlapBuilderInputType;
import crazydev.iccube.olap.loggers.OlapLoggers;
import org.bson.Document;
import org.bson.types.ObjectId;
import org.joda.time.LocalDateTime;
import org.joda.time.chrono.ISOChronology;

import java.util.*;

public abstract class OlapBuilderMongoDbAggregateExTableDecoder
{
    private OlapBuilderMongoDbAggregateExTableDecoder()
    {
    }

    public static Comparable[] decode(List<IOlapBuilderDataColumnDef> columns, Document document)
    {
        final Comparable[] values = new Comparable[columns.size()];

        for (int ii = 0; ii < columns.size(); ii++)
        {
            final IOlapBuilderDataColumnDef column = columns.get(ii);

            final String[] path = column.getName().split("\\.");

            try
            {
                final Comparable value = decodeJsonObject(document, column, path, 0);

                values[ii] = value;

            }
            catch (Exception ex)
            {
                OlapLoggers.BUILDER.error("[MongoDB] unexpected processing error", ex);
                OlapLoggers.BUILDER.error("[MongoDB] -- document : \n" + document.toJson());

                throw new OlapBuilderErrorException(ex, OlapBuilderErrorCode.UNEXPECTED_ERROR, ex.getMessage());
            }
        }

        return values;
    }

    private static Comparable decodeJsonObject(Object json, IOlapBuilderDataColumnDef column, String[] path, int pp)
            throws DecodeError
    {
        try
        {
            if (pp == path.length)
            {
                // ---------------------------------------------------------------------------------------------------------
                // We're done walking inside the nested objects: let's decode it.

                return decodeJsonObjectF(json, column, path, pp);
            }

            // -------------------------------------------------------------------------------------------------------------
            // Still walking inside nested objects.

            if (!(json instanceof Map))
            {
                throw new DecodeError("unexpected JSON non-map");
            }

            final String key = path[pp];
            final Object obj = ((Map) json).get(key);

            if (obj instanceof List)
            {
                // ---------------------------------------------------------------------------------------------------------
                // The table cell will contains the array of values (no crossjoin being performed).

                final List<Comparable> arrValues = new ArrayList<>();

                final List jsonARR = (List) obj;

                for (int ii = 0; ii < jsonARR.size(); ii++)
                {
                    final Object jsonITEM = jsonARR.get(ii);
                    final Comparable jsonVALUE = decodeJsonObject(jsonITEM, column, path, pp + 1);
                    arrValues.add(jsonVALUE);
                }

                return new CdComparableArray(arrValues);
            }

            return decodeJsonObject(obj, column, path, pp + 1);
        }
        catch (DecodeError ex)
        {
            throw new DecodeError("error while decoding JSON object " + Arrays.toString(path) + "[" + pp + "]", ex);
        }
    }

    /**
     * We're done walking inside the nested objects: let's decode it
     */
    private static Comparable decodeJsonObjectF(Object json, IOlapBuilderDataColumnDef column, String[] path, int pp)
            throws DecodeError
    {
        final Comparable value;

        if (json instanceof Map)
        {
            throw new DecodeError("unexpected JSON map " + Arrays.toString(path) + "[" + pp + "]");
        }
        else if (json != null)
        {
            value = decodeJsonRawValue(column, json);
        }
        else
        {
            value = null;
        }

        return value;
    }

    private static Comparable decodeJsonRawValue(IOlapBuilderDataColumnDef column, Object json)
            throws DecodeError
    {
        final Object iValue;

        if (json instanceof ObjectId)
        {
            iValue = ((ObjectId) json).toHexString();
        }
        else if (json instanceof String)
        {
            iValue = json;
        }
        else if (json instanceof Number)
        {
            iValue = json;
        }
        else if (json instanceof Date)
        {
            iValue = new LocalDateTime(((Date) json).getTime(), ISOChronology.getInstanceUTC());
        }
        else
        {
            throw new DecodeError("unexpected JSON object type [" + (json != null ? json.getClass().getName() : "<null>") + "]");
        }

        final OlapBuilderInputType type = column.getType();
        final IOlapBuilderInputTypeConverter converter = type.getTypeConverter();

        final Comparable value = converter.toJavaNativeValue(column.getName(), iValue);
        return value;
    }

    static class DecodeError extends Exception
    {
        private static final long serialVersionUID = 9108388412816419452L;

        public DecodeError(String message)
        {
            super(message);
        }

        public DecodeError(String message, Throwable cause)
        {
            super(message, cause);
        }
    }


}
