/*
 * Copyright 2014 - 2019 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.mongodb.datatable.aggregate;

import com.mongodb.client.DistinctIterable;
import com.mongodb.client.MongoCollection;
import crazydev.common.property.CdProperty;
import crazydev.common.property.CdReadWriteProperty;
import crazydev.common.utils.CdJson;
import crazydev.common.utils.CdStringUtils;
import crazydev.iccube.builder.OlapBuilderConnectionPool;
import crazydev.iccube.builder.OlapBuilderContext;
import crazydev.iccube.builder.datasource.reader.IOlapBuilderTablePartitionKey;
import crazydev.iccube.builder.datasource.reader.IOlapBuilderTableRowReader;
import crazydev.iccube.builder.model.def.IOlapBuilderDataSource;
import crazydev.iccube.builder.model.def.IOlapBuilderDataTableDef;
import crazydev.iccube.builder.model.validation.OlapBuilderValidator;
import crazydev.iccube.builder.mongodb.datasource.OlapBuilderMongoDbConnection;
import crazydev.iccube.builder.mongodb.datatable.common.OlapBuilderMongoDbWithPrototypeDataTable;
import jakarta.xml.bind.annotation.XmlAttribute;
import jakarta.xml.bind.annotation.XmlElement;
import jakarta.xml.bind.annotation.XmlRootElement;
import org.bson.BsonValue;
import org.bson.Document;
import org.jetbrains.annotations.Nullable;

import java.util.Objects;

@XmlRootElement(name = "mongoDB-aggregate")
public class OlapBuilderMongoDbAggregateDataTable extends OlapBuilderMongoDbWithPrototypeDataTable
{
    public static final CdProperty PIPELINE = new CdReadWriteProperty(OlapBuilderMongoDbAggregateDataTable.class, "pipeline", true)
    {
        @Override
        public Class<?> getTypeForDefaultEditor()
        {
            return CdJson.class;
        }
    };

    public static final CdProperty PIPELINE_INCR_LOAD = new CdReadWriteProperty(OlapBuilderMongoDbAggregateDataTable.class, "pipelineIncrLoad", false)
    {
        @Override
        public boolean isForIncrLoad()
        {
            return true;
        }

        @Override
        public boolean isForPartitioning()
        {
            return true;
        }

        @Override
        public Class<?> getTypeForDefaultEditor()
        {
            return CdJson.class;
        }
    };

    public static final CdProperty ALLOW_DISK_USE = new CdReadWriteProperty(GROUP_ADVANCED, OlapBuilderMongoDbAggregateDataTable.class, "allowDiskUse", false);

    public static final CdProperty BATCH_SIZE = new CdReadWriteProperty(GROUP_ADVANCED, OlapBuilderMongoDbAggregateDataTable.class, "batchSize", false);

    public static final CdProperty OPTIONS = new CdReadWriteProperty(GROUP_ADVANCED, OlapBuilderMongoDbAggregateDataTable.class, "options", false);

    public static final CdProperty PARTITIONFIELD = new CdReadWriteProperty(OlapBuilderMongoDbAggregateDataTable.class, "partitionField", false)
    {
        @Override
        public boolean isForPartitioning()
        {
            return true;
        }
    };

    @XmlAttribute(name = "allowDiskUse", required = false)
    @Nullable
    protected Boolean allowDiskUse;

    @XmlAttribute(name = "batchSize", required = false)
    @Nullable
    protected Integer batchSize;

    @XmlElement(name = "pipeline", required = true)
    private String pipeline;

    @XmlAttribute(name = "partitionField", required = false)
    private String partitionField;

    @XmlElement(name = "pipelineIncrLoad", required = false)
    private String pipelineIncrLoad;

    @XmlAttribute(name = "options", required = false)
    private String options;

    public OlapBuilderMongoDbAggregateDataTable()
    {
    }

    public OlapBuilderMongoDbAggregateDataTable(String name, String collection, @Nullable String jsonPrototype, String pipeline, @Nullable Boolean allowDiskUse, @Nullable Integer batchSize)
    {
        super(name, collection, jsonPrototype);

        this.pipeline = pipeline;
        this.allowDiskUse = allowDiskUse;
        this.batchSize = batchSize;
    }

    public String getPipeline()
    {
        return pipeline;
    }

    @Nullable
    public String getPipelineIncrLoad(@Nullable Comparable incrLoadMarker, @Nullable Object partitionValue)
    {
        final String json = addMarkers(pipelineIncrLoad, incrLoadMarker, partitionValue);
        return json;
    }

    @Nullable
    public Boolean getAllowDiskUse()
    {
        return allowDiskUse;
    }

    @Nullable
    public Integer getBatchSize()
    {
        return batchSize;
    }

    @Override
    public String getInternalFriendlyTypeName()
    {
        return "MongoDB Aggregate Table";
    }

    @Override
    public OlapBuilderValidator<IOlapBuilderDataSource<OlapBuilderMongoDbConnection>, IOlapBuilderDataTableDef<OlapBuilderMongoDbConnection>> getValidator()
    {
        return new OlapBuilderMongoDbAggregateDataTableValidator();
    }

    @Override
    protected IOlapBuilderTableRowReader<OlapBuilderMongoDbConnection> doCreateFullTableRowReader(OlapBuilderContext context,
                                                                                                  OlapBuilderConnectionPool connectionPool,
                                                                                                  int maxRowCount)
    {
        return new OlapBuilderMongoDbAggregateTableRowReader(context, connectionPool, maxRowCount, this, "MongoDB " + getName() + ".aggregate()", null);
    }

    @Override
    public boolean hasTablePartitioning()
    {
        return CdStringUtils.isNotNullAndNotBlank(partitionField);
    }

    @Override
    public @Nullable String getPartitioningColumnName()
    {
        return partitionField;
    }

    @Override
    protected IOlapBuilderTableRowReader<OlapBuilderMongoDbConnection> doCreatePartitionTableRowReaders(OlapBuilderContext context, OlapBuilderConnectionPool connectionPool, IOlapBuilderTablePartitionKey partitionKey)
    {
        return new OlapBuilderMongoDbAggregateTableRowReader(context, connectionPool, -1, this, "MongoDB " + getName() + ".find()", partitionKey);
    }

    @Override
    protected Object findOneForDiscoverColumns(OlapBuilderMongoDbConnection connection)
    {
        return OlapBuilderMongoDbAggregateTableRowReader.first(connection, this);
    }

    @Override
    public @Nullable String getOptions()
    {
        return options;
    }

    public boolean hasQueryIncrLoadPart()
    {
        return CdStringUtils.isNotNullAndNotBlank(pipelineIncrLoad);
    }

    @Override
    protected DistinctIterable<BsonValue> getDistinct(MongoCollection<Document> collection)
    {
        return collection.distinct(getPartitioningColumnName(), BsonValue.class);
    }

    @Override
    public boolean isRefreshColumnOnUpdate(IOlapBuilderDataTableDef newTable)
    {
        if (super.isRefreshColumnOnUpdate(newTable))
        {
            return true;
        }

        final OlapBuilderMongoDbAggregateDataTable tableUpdate = (OlapBuilderMongoDbAggregateDataTable) newTable;
        return !Objects.equals(this.pipeline, tableUpdate.pipeline);
    }

}
