/*
 * Copyright 2014 - 2020 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.mongodb.datasource;

import com.mongodb.MongoClientSettings;
import com.mongodb.MongoException;
import crazydev.common.exception.programming.CdShouldNotBeHereProgrammingException;
import crazydev.common.property.CdProperty;
import crazydev.common.property.CdReadWriteProperty;
import crazydev.common.security.CdPassword;
import crazydev.common.utils.CdDefaultEnumerableForUi;
import crazydev.common.utils.CdEnumerableForUi;
import crazydev.common.utils.CdProperties;
import crazydev.common.utils.CdStringUtils;
import crazydev.common.xml.CdPasswordXmlAdapter;
import crazydev.iccube.builder.errors.OlapBuilderErrorException;
import crazydev.iccube.builder.errors.OlapBuilderErrorManager;
import crazydev.iccube.builder.factory.schema.IOlapBuilderJaxbListener;
import crazydev.iccube.builder.model.def.IOlapBuilderDataSource;
import crazydev.iccube.builder.model.def.IOlapBuilderTabularDataDef;
import crazydev.iccube.builder.model.def.IOlapBuilderValidationEnabled;
import crazydev.iccube.builder.model.impl.OlapBuilderBaseDataSource;
import crazydev.iccube.builder.model.impl.table.OlapBuilderBaseDataTable;
import crazydev.iccube.builder.model.validation.OlapBuilderValidator;
import crazydev.iccube.builder.mongodb.common.OlapBuilderMongoDbHelper;
import crazydev.iccube.builder.mongodb.common.OlapBuilderMongoDbLoggers;
import crazydev.iccube.builder.mongodb.datatable.aggregate.OlapBuilderMongoDbAggregateDataTable;
import crazydev.iccube.builder.mongodb.datatable.distinct.OlapBuilderMongoDbDistinctDataTable;
import crazydev.iccube.builder.mongodb.datatable.find.OlapBuilderMongoDbFindDataTable;
import crazydev.iccube.builder.mongodb.datatable.mapreduce.OlapBuilderMongoDbMapReduceDataTable;
import crazydev.iccube.builder.mongodb.error.OlapBuilderMongoDbErrorCode;
import crazydev.iccube.builder.ux.meta.common.wizard.UxBuilderWizardStep;
import crazydev.iccube.builder.ux.meta.common.wizard.UxBuilderWizardValidation;
import crazydev.iccube.builder.ux.meta.datasource.UxBuilderDataSourceType;
import crazydev.iccube.builder.ux.meta.datasource.UxBuilderDataSourceTypeGroupId;
import crazydev.iccube.builder.ux.meta.datasource.plugin.UxBuilderMongoDbConnectionType;
import crazydev.iccube.builder.ux.meta.datasource.wizard.UxBuilderCreateDataSourceWizardSteps;
import crazydev.iccube.builder.ux.meta.datasource.wizard.UxBuilderCreateTypedDataSourceWizard;
import crazydev.iccube.builder.ux.meta.datasource.wizard.UxBuilderDataSourceNameForm;
import crazydev.iccube.olap.component.context.OlapRuntimeContext;
import crazydev.iccube.olap.loggers.OlapLoggers;
import jakarta.xml.bind.annotation.XmlAttribute;
import jakarta.xml.bind.annotation.XmlElement;
import jakarta.xml.bind.annotation.XmlRootElement;
import jakarta.xml.bind.annotation.XmlTransient;
import jakarta.xml.bind.annotation.adapters.XmlJavaTypeAdapter;
import org.jetbrains.annotations.Nullable;
import org.joda.time.CdJodaTimeUtil;
import org.joda.time.DateTimeZone;

import java.io.IOException;
import java.io.StringReader;
import java.util.*;

@XmlRootElement(name = "mongoDB-DS")
public class OlapBuilderMongoDbDataSource extends OlapBuilderBaseDataSource<OlapBuilderMongoDbConnection>
        implements IOlapBuilderJaxbListener
{
    public static final CdProperty CONNECTIONSTRING = new CdReadWriteProperty(OlapBuilderMongoDbDataSource.class, "connectionString", false);

    public static final CdProperty DB_NAME = new CdReadWriteProperty(OlapBuilderMongoDbDataSource.class, "dbName", true);

    public static final CdProperty PASSWORD = new CdReadWriteProperty(OlapBuilderMongoDbDataSource.class, "password", false)
    {
        @Override
        public Class<?> getTypeForDefaultEditor()
        {
            return CdPasswordXmlAdapter.class;
        }
    };

    public static final List<String> COMPRESSIONS = Arrays.asList("Snappy", "Zlib", "None");

    public static final CdProperty APPLY_DATE_TIME_TZ_TO_DATE = new CdReadWriteProperty(GROUP_ADVANCED, OlapBuilderMongoDbDataSource.class, "applyDateTimeTZtoDate", false);

    public static final CdProperty DATE_TIME_TZ = new CdReadWriteProperty(GROUP_ADVANCED, OlapBuilderMongoDbDataSource.class, "dateTimeTZ", false)
    {
        @Override
        public Class<?> getTypeForDefaultEditor()
        {
            return CdEnumerableForUi.class /* for now handle as an enum for the sake of simplicity in the UI side */;
        }

        @Override
        public List<String> getPossibleValues(@Nullable Object modelOwner, @Nullable Object model)
        {
            return getDateTimeTZAs();
        }

        @Override
        public Object get(Object bean)
        {
            final OlapBuilderMongoDbDataSource dataSource = (OlapBuilderMongoDbDataSource) bean;
            return dataSource.getDateTimeTZAsEnumerable();
        }
    };

    public static final CdProperty PROPERTIES = new CdReadWriteProperty(GROUP_ADVANCED, OlapBuilderMongoDbDataSource.class, "properties", false)
    {
        @Override
        public Class<?> getTypeForDefaultEditor()
        {
            return CdProperties.class;
        }
    };

    public static final CdProperty SERVER_NAME = new CdReadWriteProperty(GROUP_FORM, OlapBuilderMongoDbDataSource.class, "serverName", false);

    public static final CdProperty PORT_NUMBER = new CdReadWriteProperty(GROUP_FORM, OlapBuilderMongoDbDataSource.class, "portNumber", false);

    public static final CdProperty USER = new CdReadWriteProperty(GROUP_FORM, OlapBuilderMongoDbDataSource.class, "user", false);

    public static final CdProperty COMPRESSION = new CdReadWriteProperty(GROUP_FORM, OlapBuilderMongoDbDataSource.class, "compression", false)
    {
        @Override
        public Class<?> getTypeForDefaultEditor()
        {
            return CdEnumerableForUi.class /* for now handle as an enum for the sake of simplicity in the UI side */;
        }

        @Override
        public List<String> getPossibleValues(@Nullable Object modelOwner, @Nullable Object model)
        {
            return COMPRESSIONS;
        }

        @Override
        public Object get(Object bean)
        {
            final OlapBuilderMongoDbDataSource dataSource = (OlapBuilderMongoDbDataSource) bean;
            return new CdDefaultEnumerableForUi(dataSource.compression, COMPRESSIONS);
        }
    };

    public static final CdProperty AUTH_MECHANISM = new CdReadWriteProperty(GROUP_FORM, OlapBuilderMongoDbDataSource.class, "authMechanism", false);

    final static List<Class<? extends IOlapBuilderTabularDataDef>> CREATED_TABLE_TYPES = Arrays.asList(
            OlapBuilderMongoDbAggregateDataTable.class,
            OlapBuilderMongoDbFindDataTable.class,
            OlapBuilderMongoDbDistinctDataTable.class
    );

    @XmlTransient
    @Nullable
    private final Object cachedAllCollectionNamesLOCK = new Object();

    @XmlAttribute(required = true)
    private String serverName;

    @XmlAttribute(required = false)
    private Integer portNumber;

    @XmlAttribute(required = true)
    private String dbName;

    @XmlAttribute(required = true)
    private String compression;

    @XmlAttribute(required = false)
    @Nullable
    private OlapBuilderMongoDbAuth authMechanism;

    @XmlAttribute(required = false)
    @Nullable
    private String user;

    /**
     * [#1895] CdPassword allows for keeping the encrypted value (no re-encrypt on each save).
     */
    @XmlJavaTypeAdapter(value = CdPasswordXmlAdapter.class)
    @XmlAttribute(required = false)
    @Nullable
    private CdPassword password;

    @XmlAttribute(required = false)
    @Nullable
    private String dateTimeTZ;

    @XmlAttribute(required = false)
    private boolean applyDateTimeTZtoDate;

    @Nullable
    @XmlAttribute(required = false)
    private String connectionString;

    @XmlElement(required = false)
    @Nullable
    private String properties = OlapBuilderMongoDbHelper.getDefaultSetting();

    @XmlElement(required = false)
    @Nullable
    private String savedCollections;

    @XmlTransient
    @Nullable
    private String cachedAllCollectionNamesKey;

    @XmlTransient
    @Nullable
    private volatile List<String> cachedAllCollectionNames;

    private OlapBuilderMongoDbDataSource()
    {
    }

    public OlapBuilderMongoDbDataSource(String name)
    {
        super(name);
    }

    @Override
    public UxBuilderDataSourceTypeGroupId getUxGroupId()
    {
        return UxBuilderDataSourceTypeGroupId.dsTypeGroupNoSql;
    }

    @Override
    protected String getReportDataSourceType()
    {
        return "mongoDB";
    }

    @Override
    protected String getReportDataSourceTypeCaption()
    {
        return "MongoDB";
    }

    @Override
    public void beforeMarshal()
    {
    }

    @Override
    public void afterUnmarshal()
    {
        if (CdStringUtils.isNullOrBlank(properties))
        {
            properties = OlapBuilderMongoDbHelper.getDefaultSetting();
        }
    }

    @Nullable
    public String getCompression()
    {
        return compression;
    }

    public String getServerName()
    {
        return serverName;
    }

    public int getPortNumber()
    {
        // http://docs.mongodb.org/manual/reference/default-mongodb-port/
        return portNumber != null ? portNumber : 27017;
    }

    public String getDbName()
    {
        return dbName;
    }

    public OlapBuilderMongoDbAuth getAuthMechanism()
    {
        if (authMechanism == null)
        {
            return OlapBuilderMongoDbAuth.DEFAULT;
        }
        return authMechanism;
    }

    @Nullable
    public String getUser()
    {
        return user;
    }

    @Nullable
    public String getPassword()
    {
        return password != null ? password.clear : null;
    }

    @Nullable
    public String getProperties()
    {
        return properties;
    }

    public MongoClientSettings.Builder getOptions(MongoClientSettings.Builder optionsBuilder, int serverSelectionTimeout)
    {
        if (CdStringUtils.isNotNullAndNotBlank(properties))
        {
            try
            {
                final Properties asProperties = new Properties();

                asProperties.load(new StringReader(properties));

                for (Map.Entry<Object, Object> entry : asProperties.entrySet())
                {
                    final String name = (String) entry.getKey();
                    final int value = Integer.valueOf((String) entry.getValue());
                    OlapBuilderMongoDbHelper.applySettings(optionsBuilder, name, value); /* does nothing if not applicable */
                }
                if (serverSelectionTimeout > 0)
                {
                    OlapBuilderMongoDbHelper.applySettings(optionsBuilder, OlapBuilderMongoDbHelper.CLUSTER_SERVER_SELECTION_TIMEOUT, serverSelectionTimeout);
                }
            }
            catch (IOException | RuntimeException ex)
            {
                OlapLoggers.BUILDER.error("MongoDB connection options error", ex);
                throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.ERROR, ex.getMessage());
            }
        }
        return optionsBuilder;
    }

    @Override
    public boolean isDiscoverTablesSupported()
    {
        return false;
    }

    /**
     * @return a possibly null list of table that can be created by this datasource.
     */
    @Override
    @Nullable
    public List<Class<? extends IOlapBuilderTabularDataDef>> getCreatedTableTypes()
    {
        return CREATED_TABLE_TYPES;
    }

    @Override
    public OlapBuilderBaseDataTable<OlapBuilderMongoDbConnection> createEmptyCreateTable(@Nullable Class<IOlapBuilderTabularDataDef> tableType)
    {
        final OlapBuilderBaseDataTable<OlapBuilderMongoDbConnection> table;

        if (OlapBuilderMongoDbAggregateDataTable.class.equals(tableType))
        {
            table = new OlapBuilderMongoDbAggregateDataTable();
        }
        else if (OlapBuilderMongoDbFindDataTable.class.equals(tableType))
        {
            table = new OlapBuilderMongoDbFindDataTable();
        }
        else if (OlapBuilderMongoDbMapReduceDataTable.class.equals(tableType))
        {
            table = new OlapBuilderMongoDbMapReduceDataTable();
        }
        else if (OlapBuilderMongoDbDistinctDataTable.class.equals(tableType))
        {
            table = new OlapBuilderMongoDbDistinctDataTable();
        }
        else
        {
            throw new RuntimeException("internal error : unexpected MongoDB table type [" + (tableType != null ? tableType.getSimpleName() : "n/a") + "]");
        }

        table.setDataSource(this) /* later get-collection-names support */;

        return table;
    }

    private String getKeyForCache()
    {
        if (CdStringUtils.isNotNullAndNotBlank(connectionString))
        {
            return connectionString;
        }
        else
        {
            return "" + serverName + portNumber + dbName + user;
        }
    }

    /**
     * UI purpose.
     */
    public List<String> getCollectionNamesForUi()
    {
        final String key = getKeyForCache();

        synchronized (cachedAllCollectionNamesLOCK)
        {
            final List<String> cachedNames = cachedAllCollectionNames;
            final String cachedNamesKey = cachedAllCollectionNamesKey;

            if (cachedNames != null && cachedNamesKey != null && cachedNamesKey.equals(key))
            {
                return cachedNames;
            }
            if (CdStringUtils.isNullOrBlank(savedCollections))
            {
                final OlapBuilderMongoDbConnection conn = new OlapBuilderMongoDbConnection(this);
                try
                {
                    conn.open(null);
                    final List<String> names = conn.getCollectionNames();
                    cacheCollectionNames(getKeyForCache(), names);
                }
                finally
                {
                    conn.close();
                }
            }

            if (!savedCollections.isEmpty())
            {
                // refresh ?
                return Arrays.asList(savedCollections.split("\\."));
            }
        }
        // strange..
        return Collections.emptyList();
    }

    private void cacheCollectionNames(String key, List<String> tables)
    {
        synchronized (cachedAllCollectionNamesLOCK)
        {
            cachedAllCollectionNamesKey = key;
            cachedAllCollectionNames = tables;
            savedCollections = String.join(".", cachedAllCollectionNames);
        }
    }

    @Override
    public OlapBuilderValidator<IOlapBuilderValidationEnabled, IOlapBuilderDataSource> getValidator()
    {
        return new OlapBuilderMongoDbDataSourceValidator();
    }

    @Override
    public List<String> discoverAllTablesNames(OlapBuilderMongoDbConnection connection, boolean filterSystemSchemas, @Nullable String filter)
    {
        try
        {
            final List<String> names = connection.getCollectionNames();
            cacheCollectionNames(getKeyForCache(), names);
            return names;
        }
        catch (MongoException ex)
        {
            OlapBuilderMongoDbLoggers.GENERAL.error("[MongoDB] unexpected error", ex);
            throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.ERROR, ex.getMessage());

        }
    }

    @Override
    public OlapBuilderMongoDbConnection createConnection(OlapRuntimeContext context, boolean forceRefresh)
    {
        return createConnection();
    }

    public OlapBuilderMongoDbConnection createConnection()
    {
        return new OlapBuilderMongoDbConnection(this);
    }

    /**
     * Gives the opportunity to refresh cached entities (e.g., failed before for whatever reason).
     * Do not throw any exception around here as this method is part of a different logic.
     *
     * @param connection an opened connection (do NOT close it).
     *
     * @return
     */
    @Override
    public boolean refreshCachedEntities(OlapBuilderMongoDbConnection connection)
    {
        final List<String> tables;
        try
        {
            tables = discoverAllTablesNames(connection, true, null);
        }
        catch (Exception ignored)
        {
            return false;
        }

        final String key = getKeyForCache();
        cacheCollectionNames(key, tables);
        return true;
    }

    public boolean applyDateTimeTZtoDate()
    {
        return applyDateTimeTZtoDate;
    }

    @Nullable
    public DateTimeZone getDateTimeTZ()
    {
        if (CdStringUtils.isNullOrBlank(dateTimeTZ))
        {
            return null;
        }

        try
        {
            final DateTimeZone tz = DateTimeZone.forID(dateTimeTZ);
            return tz;

        }
        catch (IllegalArgumentException ex)
        {
            throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.ERROR, "Unexpected date timezone ID [" + dateTimeTZ + "]!");
        }
    }

    private CdEnumerableForUi getDateTimeTZAsEnumerable()
    {
        return new CdDefaultEnumerableForUi(dateTimeTZ, getDateTimeTZAs());
    }

    public static List<String> getDateTimeTZAs()
    {
        return CdJodaTimeUtil.availableIDs();
    }

    @Nullable
    public String getDecodedConnectionString()
    {
        if (CdStringUtils.isNotNullAndNotBlank(connectionString))
        {
            return connectionString.replace("<password>", password != null ? password.clear : "");
        }
        else
        {
            return null;
        }
    }

    @Override
    protected UxBuilderDataSourceType createUxType()
    {
        final String id = getClass().getSimpleName();

        return new UxBuilderDataSourceType(id, getUxTypeImage(), () -> new UxBuilderCreateTypedDataSourceWizard(id, createUxCreateWizardSteps())
        {
            /**
             * Validate the wizard models until the 'activeStep' possibly returning new steps.
             *
             * @throws OlapBuilderErrorException properly handled in the UI
             */
            public UxBuilderWizardValidation validate(OlapRuntimeContext context, int activeStep)
            {
                final UxBuilderDataSourceNameForm name = (UxBuilderDataSourceNameForm) steps.get(0).getModel();
                final UxBuilderMongoDbConnectionType type = (UxBuilderMongoDbConnectionType) steps.get(1).getModel();

                if (activeStep == 1)
                {
                    List<UxBuilderWizardStep> nsteps;
                    switch (type.getConnectionType())
                    {
                        case FormBased:
                            nsteps = doFormBased(name.getName());
                            break;
                        case ConnectionStringOnly:
                            nsteps = doConnectionString(name.getName());
                            break;
                        default:
                            throw new CdShouldNotBeHereProgrammingException();
                    }
//                    nsteps.add(UxBuilderCreateDataSourceWizardSteps.testConnection());
                    return new UxBuilderWizardValidation(activeStep, nsteps);
                }
                // validation
                if (activeStep == 2)
                {
                    final OlapBuilderMongoDbDataSource ds = (OlapBuilderMongoDbDataSource) steps.get(2).getModel();
                    if (type.getConnectionType() == UxBuilderMongoDbConnectionType.ConnectionType.ConnectionStringOnly)
                    {
                        final OlapBuilderErrorManager errorMgr = new OlapBuilderErrorManager(true);
                        OlapBuilderMongoDbDataSourceValidator.validateConnectionString(errorMgr, ds.getDecodedConnectionString());
                    }
                    return new UxBuilderWizardValidation(activeStep);
                }
                throw new CdShouldNotBeHereProgrammingException("" + activeStep);
            }

            private List<UxBuilderWizardStep> doConnectionString(String name)
            {
                final List<UxBuilderWizardStep> newSteps = new ArrayList<>();

                final OlapBuilderMongoDbDataSource dataSource = new OlapBuilderMongoDbDataSource(name);

                newSteps.add(UxBuilderCreateDataSourceWizardSteps.propsInclude(
                        true, dataSource,
                        OlapBuilderMongoDbDataSource.DB_NAME,
                        OlapBuilderMongoDbDataSource.CONNECTIONSTRING,
                        OlapBuilderMongoDbDataSource.PASSWORD
                ));

                return newSteps;
            }

            private List<UxBuilderWizardStep> doFormBased(String name)
            {
                final List<UxBuilderWizardStep> newSteps = new ArrayList<>();

                final OlapBuilderMongoDbDataSource dataSource = new OlapBuilderMongoDbDataSource(name);

                newSteps.add(UxBuilderCreateDataSourceWizardSteps.propsExclude(
                        true, dataSource,
                        OlapBuilderMongoDbDataSource.NAME,
                        OlapBuilderMongoDbDataSource.DESCRIPTION,
                        OlapBuilderMongoDbDataSource.CONNECTIONSTRING
                ));
                return newSteps;
            }

            /**
             * Restore the server side data source from edited models of the wizards.
             */
            @Override
            public IOlapBuilderDataSource getDataSourceFromUI()
            {
                final UxBuilderDataSourceNameForm name = (UxBuilderDataSourceNameForm) steps.get(0).getModel();
                final OlapBuilderMongoDbDataSource ds = (OlapBuilderMongoDbDataSource) steps.get(2).getModel();

                ds.setNameAndDescription(name.getName(), name.getDescription());

                return ds;
            }
        });
    }

    protected List<UxBuilderWizardStep> createUxCreateWizardSteps()
    {
        final List<UxBuilderWizardStep> steps = new ArrayList<>();

        steps.add(UxBuilderCreateDataSourceWizardSteps.name());

        steps.add(UxBuilderCreateDataSourceWizardSteps.create(
                "wizard.createDataSource.mongodb.connectionType", false, true, true, true, true,
                new UxBuilderMongoDbConnectionType(
                        UxBuilderMongoDbConnectionType.ConnectionType.ConnectionStringOnly
                )
        ));

//        steps.add(UxBuilderCreateDataSourceWizardSteps.testConnection());

        return steps;
    }

}
