/*
 * Copyright 1999 - 2014 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.mongodb.common;

import com.iccube.bson.decoder.ic3BsonDateTimeDecoder;
import com.mongodb.BasicDBList;
import com.mongodb.DBObject;
import com.mongodb.MongoClientSettings;
import com.mongodb.util.JSON;
import crazydev.common.exception.programming.CdUnsupportedSwitchTypeException;
import crazydev.common.utils.CdStringUtils;
import crazydev.iccube.builder.errors.OlapBuilderErrorException;
import crazydev.iccube.builder.mongodb.error.OlapBuilderMongoDbErrorCode;
import crazydev.iccube.builder.type.OlapBuilderInputType;
import crazydev.iccube.olap.loggers.OlapLoggers;
import org.bson.BsonValue;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.io.StringReader;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;

public abstract class OlapBuilderMongoDbHelper
{
    public static final String CLUSTER_SERVER_SELECTION_TIMEOUT = "cluster.serverSelectionTimeout";

    private static final Map<String, BiConsumer<MongoClientSettings.Builder, Integer>> options = new HashMap<>();

    static
    {
        // cluster
        options.put(CLUSTER_SERVER_SELECTION_TIMEOUT, (builder, value) -> builder.applyToClusterSettings(builder1 -> builder1.serverSelectionTimeout(value, TimeUnit.MILLISECONDS)));
        options.put("cluster.localThreshold", (builder, value) -> builder.applyToClusterSettings(builder1 -> builder1.localThreshold(value, TimeUnit.MILLISECONDS)));
        options.put("cluster.maxWaitQueueSize", (builder, value) -> builder.applyToClusterSettings(builder1 -> builder1.maxWaitQueueSize(value)));
        // Connection Pool
        options.put("pool.maxSize", (builder, value) -> builder.applyToConnectionPoolSettings(builder1 -> builder1.maxSize(value)));
        options.put("pool.maintenanceFrequency", (builder, value) -> builder.applyToConnectionPoolSettings(builder1 -> builder1.maintenanceFrequency(value, TimeUnit.MILLISECONDS)));
        options.put("pool.maintenanceInitialDelay", (builder, value) -> builder.applyToConnectionPoolSettings(builder1 -> builder1.maintenanceInitialDelay(value, TimeUnit.MILLISECONDS)));
        options.put("pool.maxConnectionIdleTime", (builder, value) -> builder.applyToConnectionPoolSettings(builder1 -> builder1.maxConnectionIdleTime(value, TimeUnit.MILLISECONDS)));
        options.put("pool.maxConnectionLifeTime", (builder, value) -> builder.applyToConnectionPoolSettings(builder1 -> builder1.maxConnectionLifeTime(value, TimeUnit.MILLISECONDS)));
        options.put("pool.maxWaitQueueSize", (builder, value) -> builder.applyToConnectionPoolSettings(builder1 -> builder1.maxWaitQueueSize(value)));
        options.put("pool.maxWaitTime", (builder, value) -> builder.applyToConnectionPoolSettings(builder1 -> builder1.maxWaitTime(value, TimeUnit.MILLISECONDS)));
        options.put("pool.minSize", (builder, value) -> builder.applyToConnectionPoolSettings(builder1 -> builder1.minSize(value)));
        // socket
        options.put("socket.connectTimeout", (builder, value) -> builder.applyToSocketSettings(builder1 -> builder1.connectTimeout(value, TimeUnit.MILLISECONDS)));
        options.put("socket.readTimeout", (builder, value) -> builder.applyToSocketSettings(builder1 -> builder1.readTimeout(value, TimeUnit.MILLISECONDS)));
        options.put("socket.receiveBufferSize", (builder, value) -> builder.applyToSocketSettings(builder1 -> builder1.receiveBufferSize(value)));
        options.put("socket.sendBufferSize", (builder, value) -> builder.applyToSocketSettings(builder1 -> builder1.sendBufferSize(value)));
        //server
        options.put("server.heartbeatFrequency", (builder, value) -> builder.applyToServerSettings(builder1 -> builder1.heartbeatFrequency(value, TimeUnit.MILLISECONDS)));
        options.put("server.minHeartbeatFrequency", (builder, value) -> builder.applyToServerSettings(builder1 -> builder1.minHeartbeatFrequency(value, TimeUnit.MILLISECONDS)));
        // ssl
        options.put("ssl.enabled", (builder, value) -> builder.applyToSslSettings(builder1 -> builder1.enabled(value != null && value != 0)));
        options.put("ssl.invalidHostNameAllowed", (builder, value) -> builder.applyToSslSettings(builder1 -> builder1.invalidHostNameAllowed(value != null && value != 0)));
    }

    public static void applySettings(MongoClientSettings.Builder builder, String name, int value)
    {
        options.get(name).accept(builder, value);
    }

    public static String getDefaultSetting()
    {
        StringBuilder defaultOptions = new StringBuilder();
        defaultOptions.append("# All times in milliseconds and 0 for false\n");
        defaultOptions.append("\n");

        options.keySet().stream().sorted().forEach(s -> {
            if (s.equals(CLUSTER_SERVER_SELECTION_TIMEOUT))
            {
                defaultOptions.append(s).append(" =  5000").append("\n");
            }
            else
            {
                defaultOptions.append("# ").append(s).append(" =").append("\n");
            }

        });
        return defaultOptions.toString();
    }

    public static void validate(@Nullable String properties)
    {
        if (CdStringUtils.isNullOrBlank(properties))
        {
            return;
        }

        try
        {
            final Properties asProperties = new Properties();

            asProperties.load(new StringReader(properties /* OK : isEmpty check done */));

            for (Map.Entry<Object, Object> entry : asProperties.entrySet())
            {
                final String name = (String) entry.getKey();
                final String value = (String) entry.getValue();

                final BiConsumer<MongoClientSettings.Builder, Integer> method = options.get(name);

                if (method == null)
                {
                    throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.ERROR, "unknown connection option [" + name + "]");
                }

                try
                {
                    Integer.valueOf(value);
                }
                catch (NumberFormatException ex)
                {
                    throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.ERROR, "option [" + name + "] value has to be an integer [" + value + "]");
                }
            }
        }
        catch (IOException | RuntimeException ex)
        {
            OlapLoggers.BUILDER.error("MongoDB connection options error", ex);
            throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.ERROR, ex.getMessage());
        }
    }

    @Nullable
    public static DBObject asDBObject(String kind, @Nullable String value)
    {
        final String json = CdStringUtils.isNotNullAndNotBlank(value) ? value : null;

        if (json == null)
        {
            return null;
        }

        try
        {
            @SuppressWarnings("deprecation")
            final Object bson = JSON.parse(value);
            return (DBObject) bson;

        }
        catch (RuntimeException ex)
        {
            OlapBuilderMongoDbLoggers.GENERAL.error("[MongoDB] unexpected JSON (" + kind + ") error", ex);
            throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.JSON_ERROR, kind, ex.getMessage());
        }
    }

    public static List<DBObject> asDBObjectList(String kind, @Nullable String value)
    {
        final String json = CdStringUtils.isNotNullAndNotBlank(value) ? value : null;

        if (json == null)
        {
            return new ArrayList<>();
        }

        value = value.trim();

        if (!value.startsWith("["))
        {
            value = "[" + value;
        }

        if (!value.endsWith("]"))
        {
            value = value + "]";
        }

        try
        {
            @SuppressWarnings("deprecation")
            final Object bson = JSON.parse(value);

            if (bson instanceof final BasicDBList items)
            {
                final List<DBObject> result = new ArrayList<>();

                for (Object item : items)
                {
                    if (item instanceof DBObject)
                    {
                        result.add((DBObject) item);
                    }
                    else
                    {
                        throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.JSON_ERROR, kind, "not an array of objects");
                    }
                }

                return result;
            }

            throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.JSON_ERROR, kind, "not an array of objects");
        }
        catch (RuntimeException ex)
        {
            OlapBuilderMongoDbLoggers.GENERAL.error("[MongoDB] unexpected JSON (" + kind + ") error", ex);
            throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.JSON_ERROR, kind, ex.getMessage());
        }
    }

    public static OlapBuilderInputType toType(BsonValue value)
    {
        if (value == null)
        {
            return null;
        }
        return switch (value.getBsonType())
        {
            case DOUBLE -> OlapBuilderInputType.DOUBLE;
            case STRING -> OlapBuilderInputType.STRING;
            case BOOLEAN -> OlapBuilderInputType.BOOLEAN;
            case TIMESTAMP -> OlapBuilderInputType.LONG;
            case DATE_TIME -> OlapBuilderInputType.DATETIME;
            case NULL -> OlapBuilderInputType.STRING;
            case INT32 -> OlapBuilderInputType.INTEGER;
            case INT64 -> OlapBuilderInputType.LONG;
            case DECIMAL128 -> OlapBuilderInputType.DOUBLE;
            default -> null;
        };
    }

    public static Comparable toValue(OlapBuilderInputType type, BsonValue value)
    {
        if (value == null)
        {
            return null;
        }
        return switch (type)
        {
            case DOUBLE -> value.asDouble().doubleValue();
            case INTEGER -> value.asInt32().intValue();
            case STRING -> value.asString().getValue();
            case DATETIME, DATE -> ic3BsonDateTimeDecoder.toJodaDateTime(value.asDateTime().getValue());
            case BOOLEAN -> value.asBoolean().getValue();
            case FLOAT -> value.asDouble().doubleValue();
            case SHORT -> value.asInt32().intValue();
            case LONG -> value.asInt64().longValue();
            case UPPERCASE_STRING -> value.asString().getValue().toUpperCase();
            default -> throw new CdUnsupportedSwitchTypeException(type);
        };
    }

    // public static Comparable toNativeValue(BsonValue value)
    // {
    //     if (value == null)
    //     {
    //         return null;
    //     }
    //     return switch (value.getBsonType())
    //     {
    //         case DOUBLE -> value.asDouble().getValue();
    //         case STRING -> value.asString().getValue();
    //         case BOOLEAN -> value.asBoolean().getValue();
    //         case TIMESTAMP -> value.asTimestamp().getValue();
    //         case DATE_TIME -> ic3BsonDateTimeDecoder.toJodaDateTime(value.asDateTime().getValue());
    //         case NULL -> null;
    //         case INT32 -> value.asInt32().getValue();
    //         case INT64 -> value.asInt64().getValue();
    //         default -> throw new CdUnsupportedSwitchTypeException(value.getBsonType());
    //     };
    // }

}
