/*
 * Copyright 2014 - 2019 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package com.iccube.bson;

import org.bson.types.ObjectId;

import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Map;

public class ic3BsonBytesBuffer
{
    public final byte[] bytes;

    private final Map<ic3BsonStringBuilder, String> cacheCString;

    private final boolean cacheStringValues;

    public int pos;

    public ic3BsonBytesBuffer(Map<ic3BsonStringBuilder, String> cacheCString, byte[] bytes, boolean cacheStringValues)
    {
        this.cacheCString = cacheCString;
        this.bytes = bytes;
        this.pos = 0;
        this.cacheStringValues = cacheStringValues;
    }

    static private int makeInt(byte b0, byte b1, byte b2, byte b3)
    {
        return (
                ((b0 & 0xff)) |
                ((b1 & 0xff) << 8) |
                ((b2 & 0xff) << 16) |
                ((b3) << 24)
        );
    }

    static private long makeLong(byte b0, byte b1, byte b2, byte b3,
                                 byte b4, byte b5, byte b6, byte b7)
    {
        return (
                (((long) b7) << 56) |
                (((long) b6 & 0xff) << 48) |
                (((long) b5 & 0xff) << 40) |
                (((long) b4 & 0xff) << 32) |
                (((long) b3 & 0xff) << 24) |
                (((long) b2 & 0xff) << 16) |
                (((long) b1 & 0xff) << 8) |
                (((long) b0 & 0xff))
        );
    }

    public com.iccube.bson.ic3BsonType getType()
    {
        return com.iccube.bson.ic3BsonType.findExistingType((int) bytes[pos++]);
    }

    public String getFieldName()
    {
        final ic3BsonStringBuilder.StringAndLength ret = ic3BsonStringBuilder.getOrCreateString(cacheCString, bytes, this.pos);
        pos += ret.bytesLength + 1/* 0 character */;
        return ret.str;
    }

    public String getCName()
    {
        int pos = this.pos;

        while (bytes[++pos] != 0)
        {
        }

        // not cached !
        return createCString(this.pos, pos - this.pos);
    }

    public String getString()
    {
        int size = getInt();
        // cacheValue ?
        String name = createCString(pos, size - 1);
        pos += size;
        return name;
    }

    private String createCString(int initialPos, int fSize)
    {
        if (cacheStringValues)
        {
            return ic3BsonStringBuilder.getOrCreateString(cacheCString, bytes, initialPos).str;
        }
        else
        {
            byte[] nbytes = new byte[fSize];
            System.arraycopy(bytes, initialPos, nbytes, 0, fSize);

            String fieldName = new String(nbytes, StandardCharsets.UTF_8);
            return fieldName;
        }
    }

    public int getInt()
    {
        return makeInt(bytes[pos++], bytes[pos++], bytes[pos++], bytes[pos++]);
    }

    public long getLong()
    {
        return makeLong(bytes[pos++], bytes[pos++], bytes[pos++], bytes[pos++], bytes[pos++], bytes[pos++], bytes[pos++], bytes[pos++]);
    }

    public double getDouble()
    {
        return Double.longBitsToDouble(getLong());
    }

    public boolean getBoolean()
    {
        return bytes[pos++] != 0;
    }

    public void skip(int byteNumber)
    {
        pos += byteNumber;

    }

    public void skipFieldName()
    {
        while (bytes[pos++] != 0)
        {
        }
    }

    public ObjectId getObjectId()
    {
        ByteBuffer buffer = ByteBuffer.wrap(bytes, pos, 12);
        pos += 12;
        return new ObjectId(buffer);
    }

    public byte[] slice()
    {
        return Arrays.copyOfRange(bytes, pos, bytes.length);
    }
}
