/*
 * Copyright 1999 - 2016 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.plugin.olapfunctions.others;

import crazydev.common.collection.CdCollections;
import crazydev.iccube.olap.entity.OlapEntity;
import crazydev.iccube.olap.entity.scalar.OlapDoubleVectorEntity;
import crazydev.iccube.olap.entity.scalar.OlapNumericEntity;
import crazydev.iccube.olap.entity.scalar.OlapScalarEntity;
import crazydev.iccube.olap.entity.set.OlapTupleSet;
import crazydev.iccube.olap.entity.tuple.OlapTuple;
import crazydev.iccube.olap.eval.execinstr.gf.context.GFContext;
import crazydev.iccube.olap.eval.execinstr.gf.function.GFFunctionArgs;
import crazydev.iccube.olap.eval.function.OlapFunctionArgs;
import crazydev.iccube.olap.eval.function.mdx.numeric.OlapBaseGenericStatisticalFunction;
import crazydev.iccube.olap.eval.function.mdx.numeric.calculators.OlapSetCalculatorOneGenExpression;
import it.unimi.dsi.fastutil.doubles.DoubleArrayList;

import java.util.ArrayList;
import java.util.Arrays;

public class OlapVectorFunction extends OlapBaseGenericStatisticalFunction
{
    public static final String NAME = "Vector";

    public static final OlapFunctionArgs ARGS = new OlapFunctionArgs(2, 4)
    {
        @Override
        public boolean isLambda(int pos)
        {
            return pos == 1;
        }
    };

    public OlapVectorFunction()
    {
        super(NAME, ARGS);
    }

    @Override
    protected OlapSetCalculatorOneGenExpression createCalculator(GFFunctionArgs args, OlapTupleSet set)
    {
        final boolean includeEmpty = getIncludeEmpty(args);

        if (set.isUnorder())
        {
            return withoutKeepingOrder(args, includeEmpty);
        }

        if (includeEmpty)
        {
            return withEmptyKeepingOrder(args);
        }
        else
        {
            return withoutEmptyKeepingOrder(args);
        }
    }

    private OlapSetCalculatorOneGenExpression withoutKeepingOrder(GFFunctionArgs args, final boolean includeEmpty)
    {
        return new OlapSetCalculatorOneGenExpression()
        {
            final DoubleArrayList values = new DoubleArrayList();

            @Override
            public int stopIterationOrdinal()
            {
                return -1;
            }

            @Override
            public void addNull(OlapTuple setTuple, int iTupleOrdinal)
            {
                if (includeEmpty)
                {
                    values.add(0.0);
                }
            }

            @Override
            public void addValue(GFContext context, OlapTuple setTuple, int iTupleOrdinal, OlapScalarEntity value)
            {
                if (value instanceof OlapNumericEntity)
                {
                    OlapNumericEntity numeric = (OlapNumericEntity) value;
                    values.add(numeric.doubleValue());
                }
                else
                {
                    onFunctionArgTypeMismatchError(args.getErrorContext(context), 2, "numeric", /* !arg */ value.getFriendlyTypeName());
                }
            }

            @Override
            public boolean isResultEmpty()
            {
                return values.isEmpty();
            }

            @Override
            public OlapEntity calculate()
            {
                return new OlapDoubleVectorEntity(values.toDoubleArray());
            }
        };
    }

    private OlapSetCalculatorOneGenExpression withEmptyKeepingOrder(final GFFunctionArgs args)
    {
        return new OlapSetCalculatorOneGenExpression()
        {
            final DoubleArrayList values = new DoubleArrayList();

            @Override
            public int stopIterationOrdinal()
            {
                return -1;
            }

            @Override
            public void addNull(OlapTuple setTuple, int iTupleOrdinal)
            {
                CdCollections.setValue(values, iTupleOrdinal - 1, 0.0);
            }

            @Override
            public void addValue(GFContext context, OlapTuple setTuple, int iTupleOrdinal, OlapScalarEntity value)
            {
                if (value instanceof OlapNumericEntity)
                {
                    OlapNumericEntity numeric = (OlapNumericEntity) value;
                    CdCollections.setValue(values, iTupleOrdinal - 1, numeric.doubleValue());
                }
                else
                {
                    onFunctionArgTypeMismatchError(args.getErrorContext(context), 2, "numeric", /* !arg */ value.getFriendlyTypeName());
                }
            }

            @Override
            public boolean isResultEmpty()
            {
                return values.isEmpty();
            }

            @Override
            public OlapEntity calculate()
            {
                return new OlapDoubleVectorEntity(values.toDoubleArray());
            }
        };
    }

    private OlapSetCalculatorOneGenExpression withoutEmptyKeepingOrder(final GFFunctionArgs args)
    {
        return new OlapSetCalculatorOneGenExpression()
        {
            ArrayList<MyClass> list = new ArrayList<>();

            @Override
            public int stopIterationOrdinal()
            {
                return -1;
            }

            @Override
            public void addNull(OlapTuple setTuple, int iTupleOrdinal)
            {
            }

            @Override
            public void addValue(GFContext context, OlapTuple setTuple, int iTupleOrdinal, OlapScalarEntity value)
            {
                if (value instanceof OlapNumericEntity)
                {
                    OlapNumericEntity numeric = (OlapNumericEntity) value;
                    list.add(new MyClass(iTupleOrdinal, numeric.doubleValue()));
                }
                else
                {
                    onFunctionArgTypeMismatchError(args.getErrorContext(context), 2, "numeric", /* !arg */ value.getFriendlyTypeName());
                }
            }

            @Override
            public boolean isResultEmpty()
            {
                return list.isEmpty();
            }

            @Override
            public OlapEntity calculate()
            {
                //TODO (dad) J8  -> use list.sort()
                final MyClass[] array = list.toArray(new MyClass[list.size()]);
                Arrays.sort(array);

                double values[] = new double[list.size()];

                for (int i = 0; i < array.length; i++)
                {
                    final MyClass myClass = array[i];
                    values[i] = myClass.value;
                }

                return new OlapDoubleVectorEntity(values);
            }
        };
    }

    private class MyClass implements Comparable<MyClass>
    {
        int ordinal;

        double value;

        public MyClass(int ordinal, double value)
        {
            this.ordinal = ordinal;
            this.value = value;
        }

        @Override
        public int compareTo(MyClass o)
        {
            return Integer.compare(ordinal, o.ordinal);
        }
    }

}