/*
 * Copyright 1999 - 2014 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.plugin.olapfunctions.others;

import crazydev.iccube.exception.OlapErrorCode;
import crazydev.iccube.olap.entity.OlapEntity;
import crazydev.iccube.olap.entity.scalar.*;
import crazydev.iccube.olap.entity.set.OlapListTupleSet;
import crazydev.iccube.olap.entity.set.OlapTupleSet;
import crazydev.iccube.olap.entity.tuple.OlapTuple;
import crazydev.iccube.olap.eval.execinstr.gf.context.GFContext;
import crazydev.iccube.olap.eval.execinstr.gf.function.GFFunctionArgs;
import crazydev.iccube.olap.eval.function.OlapFunctionArgs;
import crazydev.iccube.olap.eval.function.OlapFunctionArgumentType;
import crazydev.iccube.olap.eval.function.OlapScalarEntityFunction;

public class OlapSetAxesFunction extends OlapScalarEntityFunction
{
    public static final String NAME = "SetAxes";

    public static final OlapFunctionArgs ARGS = new OlapFunctionArgs(2, 3)
    {
        @Override
        public OlapFunctionArgumentType getType(int pos)
        {
            switch (pos)
            {
                case 0:
                    return OlapFunctionArgumentType.VALUE;
                default:
                    return super.getType(pos);
            }
        }
    };

    public OlapSetAxesFunction()
    {
        super(NAME, ARGS);
    }

    @Override
    public OlapScalarEntity evalForCluster(GFContext context, GFFunctionArgs args)
    {
        final OlapEntity firstParam = args.toValueIfApplicable(context, 0, "matrix/vector");

        if (firstParam instanceof OlapDoubleMatrixEntity)
        {
            return fromMatrix(context, args, (OlapDoubleMatrixEntity) firstParam);
        }

        if (firstParam instanceof OlapDoubleVectorEntity)
        {
            if (args.size() != 2)
            {
                return args.onFunctionArgSizeError(context);
            }

            return fromVector(context, args, (OlapDoubleVectorEntity) firstParam);
        }

        final OlapEntity arg = args.getNonMissingArgEntity(context, 0, "matrix/vector");
        return args.onFunctionArgTypeMismatchError(context, 0, "matrix/vector", arg.getFriendlyTypeName());
    }

    private OlapScalarEntity fromVector(GFContext context, GFFunctionArgs args, OlapDoubleVectorEntity vectorEntity)
    {
        final OlapTupleSet<? extends OlapTuple> rows = args.toSet(context, 1);

        final OlapListTupleSet<?> rowList = rows.asTupleList();

        if (rowList.size() != vectorEntity.length())
        {
            return args.onFunctionError(context, OlapErrorCode.VECTOR_ERROR, "Set size '" + rowList.size() + "' does not match Vector size '" + vectorEntity.length() + "'");
        }

        return new OlapDoubleVectorAEntity(rowList, vectorEntity);
    }

    private OlapScalarEntity fromMatrix(GFContext context, GFFunctionArgs args, OlapDoubleMatrixEntity matrix)
    {
        final int colsIdx = 1;
        final int rowsIdx = 2;

        final OlapTupleSet<? extends OlapTuple> cols = args.toSet(context, colsIdx);
        final OlapTupleSet<? extends OlapTuple> rows = args.toSet(context, rowsIdx);

        final OlapListTupleSet<?> colList = cols.asTupleList();
        final OlapListTupleSet<?> rowList = rows.asTupleList();

        if (colList.size() != matrix.cols())
        {
            return args.onFunctionError(context, OlapErrorCode.MATRIX_ERROR, "First set - cols - size '" + colList.size() + "' does not match matrix column size '" + matrix.cols() + "'");
        }

        if (rowList.size() != matrix.rows())
        {
            return args.onFunctionError(context, OlapErrorCode.MATRIX_ERROR, "Second set - rows - size '" + rowList.size() + "' does not match matrix row size '" + matrix.rows() + "'");
        }

        return new OlapDoubleMatrixAEntity(colList, rowList, matrix);
    }

}