/*
 * Copyright 2014 - 2019 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.plugin.olapfunctions.others;

import crazydev.common.utils.CdTimeUtils;
import crazydev.iccube.builder.type.OlapBuilderInputType;
import crazydev.iccube.enums.OlapLevelType;
import crazydev.iccube.exception.OlapErrorCode;
import crazydev.iccube.olap.entity.level.OlapLevel;
import crazydev.iccube.olap.entity.member.OlapMember;
import crazydev.iccube.olap.entity.permissions.IOlapSchemaPermission;
import crazydev.iccube.olap.entity.scalar.OlapNonScalarEntity;
import crazydev.iccube.olap.entity.scalar.OlapScalarEntity;
import crazydev.iccube.olap.eval.exception.OlapFunctionEvaluationException;
import crazydev.iccube.olap.eval.execinstr.gf.context.GFContext;
import crazydev.iccube.olap.eval.execinstr.gf.function.GFFunctionArgs;
import crazydev.iccube.olap.eval.filter.dimension.OlapMemberFilter;
import crazydev.iccube.olap.eval.function.OlapFunctionArgs;
import crazydev.iccube.olap.eval.function.mdx.member.OlapMdxNavigationFunction;
import org.apache.commons.lang3.mutable.MutableObject;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.LocalTime;

public class OlapLookupDateFunction extends OlapMdxNavigationFunction
{
    public static final String NAME = "LookupDate";

    public static final OlapFunctionArgs ARGS = new OlapFunctionArgs(2);

    public OlapLookupDateFunction()
    {
        super(NAME, ARGS);
    }

    @Override
    public OlapNonScalarEntity evalForCluster(GFContext context, GFFunctionArgs args)
    {
        OlapLevel level = args.toLevel(context, 0);

        final OlapLevelType levelType = checkLevelIsDate(context, args, level, getName());

        final OlapScalarEntity value = args.toValue(context, 1, "scalar");
        final LocalDate lookupDate = toLocalDate(context, args, value, getName());
        if (lookupDate == null)
        {
            return asNull();
        }

        final IOlapSchemaPermission schemaPermissions = args.getSchemaPermissionsWithoutPerspective(context);
        final OlapMemberFilter memberFilter = schemaPermissions.getDimensionsPermissions(args.getCube(context)).getMemberFilter();

        MutableObject foundMember = new MutableObject();
        level.forEachMemberWithoutAllStoppable(member -> {
            if (memberFilter != null && !memberFilter.isAccepting(member))
            {
                return true;
            }
            final Comparable keyValue = member.getKeyValue();
            final LocalDate start;
            if (keyValue instanceof LocalDate)
            {
                start = (LocalDate) keyValue;
            }
            else if (keyValue instanceof LocalDateTime)
            {
                start = ((LocalDateTime) keyValue).toLocalDate();
            }
            else
            {
                return true; // unknown member ?
            }

            final boolean noMatch;
            if (levelType == OlapLevelType.DAY)
            {
                noMatch = !lookupDate.equals(start);
            }
            else
            {
                final LocalDate end = levelType.getNextDate(start.toLocalDateTime(LocalTime.MIDNIGHT)).toLocalDate();
                noMatch = lookupDate.isBefore(start) || CdTimeUtils.isAfterOrEquals(lookupDate, end);
            }

            if (noMatch)
            {
                // continue
                return true;
            }
            else
            {
                foundMember.setValue(member);
                return false;
            }
        });

        return asNullable((OlapMember) foundMember.get());
    }

    public static LocalDate toLocalDate(GFContext context, GFFunctionArgs args, OlapScalarEntity value, String functionName)
    {
        if (value.isMdxNull()) {
            return null;
        }
        final LocalDate lookupDate;
        if (value.isConvertibleToJavaNativeValueType(LocalDate.class))
        {
            lookupDate = (LocalDate) value.asValue();
        }
        else if (value.isConvertibleToJavaNativeValueType(LocalDateTime.class))
        {
            lookupDate = ((LocalDateTime) value.asValue()).toLocalDate();
        }
        else
        {
            throw new OlapFunctionEvaluationException(args.getErrorContext(context, 0), functionName, OlapErrorCode.FUNCTION_ARGUMENT_TYPE_MISMATCH, "date value", value.getClass().getSimpleName());
        }
        return lookupDate;
    }

    public static OlapLevelType checkLevelIsDate(GFContext context, GFFunctionArgs args, OlapLevel level, String functionName)
    {
        final OlapBuilderInputType[] keyType = level.getMemberKeyType();
        if (keyType == null || keyType.length != 1)
        {
            throw new OlapFunctionEvaluationException(args.getErrorContext(context, 0), functionName, OlapErrorCode.FUNCTION_ARGUMENT_TYPE_MISMATCH, "level-date-key", "array");
        }
        if (!keyType[0].isDateType())
        {
            throw new OlapFunctionEvaluationException(args.getErrorContext(context, 0), functionName, OlapErrorCode.FUNCTION_ARGUMENT_TYPE_MISMATCH, "level-date-key", keyType[0].getShortName());
        }

        return level.getLevelType() == null || level.getLevelType().hasTimeInformation() ? OlapLevelType.DAY : level.getLevelType();
    }
}