/*
 * Copyright 2014 - 2019 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.plugin.olapfunctions.others;

import crazydev.iccube.builder.type.OlapBuilderInputType;
import crazydev.iccube.collection.OlapIterator;
import crazydev.iccube.olap.entity.OlapEntity;
import crazydev.iccube.olap.entity.dimension.OlapDimension;
import crazydev.iccube.olap.entity.hierarchy.OlapHierarchy;
import crazydev.iccube.olap.entity.level.OlapLevel;
import crazydev.iccube.olap.entity.member.OlapCalculatedMember;
import crazydev.iccube.olap.entity.member.OlapMeasureMember;
import crazydev.iccube.olap.entity.member.OlapMember;
import crazydev.iccube.olap.entity.member.OlapMemberKey;
import crazydev.iccube.olap.entity.option.OlapOption;
import crazydev.iccube.olap.entity.permissions.IOlapSchemaPermission;
import crazydev.iccube.olap.entity.scalar.OlapNonScalarEntity;
import crazydev.iccube.olap.entity.scalar.OlapScalarEntity;
import crazydev.iccube.olap.entity.set.OlapSetFactory;
import crazydev.iccube.olap.entity.set.OlapTupleSet;
import crazydev.iccube.olap.entity.tuple.OlapTuple;
import crazydev.iccube.olap.eval.execinstr.gf.context.GFContext;
import crazydev.iccube.olap.eval.execinstr.gf.function.GFFunctionArgs;
import crazydev.iccube.olap.eval.filter.dimension.OlapMemberFilter;
import crazydev.iccube.olap.eval.function.OlapFunctionArgs;
import crazydev.iccube.olap.eval.function.OlapFunctionArgumentType;
import crazydev.iccube.olap.eval.function.mdx.member.OlapMdxNavigationFunction;
import org.jetbrains.annotations.Nullable;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.List;

public class OlapLookupByKeyFunction extends OlapMdxNavigationFunction
{
    public static final String NAME = "LookupByKey";

    public static final OlapFunctionArgs ARGS = new OlapFunctionArgs(2, OlapFunctionArgs.N)
    {
        @Override
        public OlapFunctionArgumentType getType(int pos)
        {
            switch (pos)
            {
                default:
                    return super.getType(pos);
            }
        }
    };

    public OlapLookupByKeyFunction()
    {
        super(NAME, ARGS);
    }

    private static OlapMemberKey[] createKey(List<OlapScalarEntity> scalars, OlapDimension dimension)
    {
        if (scalars.size() == 1)
        {
            return new OlapMemberKey[]{createSingleKey(scalars.get(0), dimension)};
        }

        return new OlapMemberKey[]{createMultiKey(scalars, dimension)};
    }

    static OlapMemberKey createSingleKey(OlapScalarEntity scalar, OlapDimension dimension)
    {
        final OlapBuilderInputType inputType = OlapBuilderInputType.findByXmlaType(scalar.getXmlaType(null));
        final Comparable convertedKeyValue = dimension.adaptKeyValue((Comparable) scalar.asValue());

        return OlapMemberKey.create(inputType, convertedKeyValue);
    }

    static OlapMemberKey createMultiKey(List<OlapScalarEntity> scalars, OlapDimension dimension)
    {
        final OlapBuilderInputType[] inputTypes = new OlapBuilderInputType[scalars.size()];
        final Comparable[] convertedKeyValues = new Comparable[scalars.size()];

        for (int idx = 0; idx < scalars.size(); idx++)
        {
            final OlapScalarEntity scalar = scalars.get(idx);

            final OlapBuilderInputType inputType = OlapBuilderInputType.findByXmlaType(scalar.getXmlaType(null));
            final Comparable convertedKeyValue = dimension.adaptKeyValue((Comparable) scalar.asValue());

            inputTypes[idx] = inputType;
            convertedKeyValues[idx] = convertedKeyValue;
        }

        return OlapMemberKey.create(inputTypes, convertedKeyValues);
    }

    @Override
    public OlapNonScalarEntity evalForCluster(GFContext context, GFFunctionArgs args)
    {
        final OlapEntity arg0 = args.getNonMissingArgEntity(context, 0, "hierarchy|dimension|level");

        final boolean typed = args.size() > 2 && args.toOptionalOptionX(context, args.size() - 1, OlapOption.TYPED) == OlapOption.TYPED;

        boolean isSet = false;

        final OlapHierarchy hierarchy;
        OlapLevel level = null;

        if (arg0 instanceof OlapDimension)
        {
            hierarchy = args.toOnlyHierarchy(context, 0, (OlapDimension) arg0);
        }
        else if (arg0 instanceof OlapHierarchy)
        {
            hierarchy = (OlapHierarchy) arg0;
        }
        else if (arg0 instanceof OlapLevel)
        {
            level = (OlapLevel) arg0;
            hierarchy = level.getHierarchy();
        }
        else
        {
            return args.onFunctionArgTypeMismatchError(context, 0, "hierarchy|dimension|level", arg0.getFriendlyTypeName());
        }

        OlapMemberKey[] key = null;

        final OlapEntity sec = args.getNonMissingArgEntity(context, 1, "member");

        if (sec instanceof OlapMeasureMember || sec instanceof OlapCalculatedMember)
        {
            final OlapScalarEntity value = args.toValue(context, 1, "scalar");
            if (value.isMdxNull())
            {
                return asNull();
            }
            key = createKey(List.of(value), hierarchy.getDimension());
        }
        else if (sec instanceof OlapMember)
        {
            key = new OlapMemberKey[]{((OlapMember) sec).getKey()};
        }
        else
        {
            final OlapTupleSet<? extends OlapTuple> toSet = args.toSetIfApplicable(context, 1);

            if (toSet != null)
            {
                final List<OlapMemberKey> keys = new ArrayList<>();

                final OlapIterator<? extends OlapTuple> tuples = toSet.getIterator(false);

                OlapTuple tuple;

                tuples.reset();

                while ((tuple = tuples.next()) != null)
                {
                    if (tuple instanceof OlapMember)
                    {
                        OlapMember member = (OlapMember) tuple;
                        keys.add(member.getKey());
                    }
                    else
                    {
                        return args.onFunctionArgTypeMismatchError(context, 0, "member-set", arg0.getFriendlyTypeName());
                    }
                }

                key = keys.toArray(new OlapMemberKey[keys.size()]);

                isSet = true;
            }
        }

        if (key == null)
        {
            key = createKey(getKeys(context, args, typed), hierarchy.getDimension());
        }

        return lookupByKey(context, args, hierarchy, level, key, typed, isSet);
    }

    private static List<OlapScalarEntity> getKeys(GFContext context, GFFunctionArgs args, boolean typed)
    {
        final List<OlapScalarEntity> keys = new ArrayList<>();

        final int size = args.size() - (typed ? 1 : 0);

        for (int idx = 1; idx < size; idx++)
        {
            final OlapScalarEntity key = args.asScalar(context, idx);
            keys.add(key);
        }

        return keys;
    }

    static OlapNonScalarEntity lookupByKey(GFContext context, GFFunctionArgs args, OlapHierarchy hierarchy, @Nullable OlapLevel level, OlapMemberKey[] key, boolean typed, boolean asSet)
    {
        if (!asSet)
        {
            return lookupByKey(context, args, hierarchy, level, key[0], typed);
        }

        final List<OlapMember> members = new ArrayList<>();

        for (int ii = 0; ii < key.length; ii++)
        {
            final OlapEntity member = lookupByKey(context, args, hierarchy, level, key[ii], typed);

            if (member instanceof OlapMember)
            {
                members.add((OlapMember) member);
            }
        }

        return OlapSetFactory.instance(members);
    }

    static OlapNonScalarEntity lookupByKey(GFContext context, GFFunctionArgs args, OlapHierarchy hierarchy, @Nullable OlapLevel olapLevel, OlapMemberKey key, boolean typed)
    {
        final OlapMember[] foundMember = new OlapMember[1];
        final Object keyValue = key.asJavaNativeValue();

        final IOlapSchemaPermission schemaPermissions = args.getSchemaPermissionsWithoutPerspective(context);
        final OlapMemberFilter memberFilter = schemaPermissions.getDimensionsPermissions(args.getCube(context)).getMemberFilter();

        if (hierarchy.canIndexByRange())
        {
            Object actualKeyValue = convertLevelForKey(hierarchy.getLastLevel(), keyValue, typed);
            if (actualKeyValue == null)
            {
                return asNull();
            }
            OlapMember[] members = hierarchy.lookupMemberByRange(olapLevel, actualKeyValue);
            if (members.length != 1)
            {
                return asNull();
            }
            foundMember[0] = getFilteredMember(members[0], memberFilter);
        }
        else
        {
            // lookup by key == lookup by id
            for (OlapLevel level : hierarchy.getLevels())
            {
                if ((olapLevel == null || olapLevel == level) && !level.isAll())
                {
                    Object actualKeyValue = convertLevelForKey(level, keyValue, typed);
                    if (actualKeyValue != null)
                    {
                        level.forEachMatchingKeyValue(args.getErrorContext(context), actualKeyValue, olapMember -> {
                            if (memberFilter == null || memberFilter.isAccepting(olapMember))
                            {
                                foundMember[0] = olapMember;
                                return false;
                            }
                            return true;
                        });
                    }
                }

                if (foundMember[0] != null)
                {
                    break;
                }
            }
        }
        return asNullable(foundMember[0]);
    }

    @Nullable
    private static OlapMember getFilteredMember(final OlapMember memberToFilter, OlapMemberFilter memberFilter)
    {
        OlapMember member = memberToFilter;
        if (member != null && memberFilter != null && !memberFilter.isAccepting(member))
        {
            // if member is filtered by the security get previous one, the range works like this
            member = member.getPrevMember(memberFilter);
            if (member == null)
            {
                member = memberToFilter.getNextMember(memberFilter);
            }
        }
        return member;
    }

    @Nullable
    private static Object convertLevelForKey(OlapLevel level, Object keyValue, boolean typed)
    {
        if (typed)
        {
            return keyValue;
        }

        final OlapBuilderInputType[] keyType = level.getMemberKeyType();

        if (keyType == null)
        {
            return null;
        }

        try
        {
            if (keyValue.getClass().isArray())
            {
                if (keyType.length != Array.getLength(keyValue))
                {
                    return null;
                }

                final Comparable[] values = new Comparable[keyType.length];

                for (int i = 0; i < keyType.length; i++)
                {
                    final OlapBuilderInputType type = keyType[i];
                    values[i] = type.getTypeConverter().toJavaNativeValue("LookupByKey", Array.get(keyValue, i));
                }

                return values;
            }
            else
            {
                return keyType[0].getTypeConverter().toJavaNativeValue("LookupByKey", keyValue);
            }
        }
        catch (Exception ex)
        {
            return null;
        }
    }


}