/*
 * Copyright 1999 - 2014 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.plugin.olapfunctions.excel;

import crazydev.common.exception.programming.CdShouldNotBeHereProgrammingException;
import crazydev.iccube.plugin.function.OlapPluginEvaluationException;
import crazydev.iccube.plugin.function.OlapPluginUserJavaFunction;
import org.jetbrains.annotations.Nullable;

@SuppressWarnings({"UnusedDeclaration"})
public abstract class OlapVBExcelTextFunctions extends OlapPluginUserJavaFunction
{
    protected OlapVBExcelTextFunctions(int minArgs, Class... parametersType)
    {
        super(minArgs, parametersType);
    }

    protected OlapVBExcelTextFunctions(int minArgs, int maxArgs, Class... parametersType)
    {
        super(minArgs, maxArgs, parametersType);
    }

    protected OlapVBExcelTextFunctions(Class[]... params)
    {
        super(params);
    }

    /* ---------------------------------------------------------------------- */

    private static abstract class SingleArgTextFunction extends OlapVBExcelTextFunctions
    {
        protected SingleArgTextFunction()
        {
            super(1, String.class);
        }

        @Override
        protected Object doEval(Object[] args) throws OlapPluginEvaluationException
        {
            return evaluate((String) args[0]);
        }

        protected abstract Object evaluate(@Nullable String arg);
    }

    public static final OlapPluginUserJavaFunction LEN = new SingleArgTextFunction()
    {
        @Override
        protected Integer evaluate(String arg)
        {
            if (arg == null)
            {
                return null /* SSAS */;
            }

            return arg.length();
        }
    };

    public static final OlapPluginUserJavaFunction LOWER = new SingleArgTextFunction()
    {
        @Override
        protected String evaluate(String arg)
        {
            if (arg == null)
            {
                return null /* SSAS : does not exist but keep it consistent with the others ... */;
            }

            return arg.toLowerCase();
        }
    };

    public static final OlapPluginUserJavaFunction LCASE = new SingleArgTextFunction()
    {
        @Override
        protected String evaluate(String arg)
        {
            if (arg == null)
            {
                return null /* SSAS : does not exist but keep it consistent with the others ... */;
            }

            return arg.toLowerCase();
        }
    };

    public static final OlapPluginUserJavaFunction UPPER = new SingleArgTextFunction()
    {
        @Override
        protected String evaluate(String arg)
        {
            if (arg == null)
            {
                return null /* SSAS : does not exist but keep it consistent with the others ... */;
            }

            return arg.toUpperCase();
        }
    };

    public static final OlapPluginUserJavaFunction UCASE = new SingleArgTextFunction()
    {
        @Override
        protected String evaluate(String arg)
        {
            if (arg == null)
            {
                return null /* SSAS : does not exist but keep it consistent with the others ... */;
            }

            return arg.toUpperCase();
        }
    };

    /**
     * An implementation of the TRIM function:
     * Removes leading and trailing spaces from value if evaluated operand
     * value is string.
     *
     * @author Manda Wilson &lt; wilson at c bio dot msk cc dot org &gt;
     */
    public static final OlapPluginUserJavaFunction TRIM = new SingleArgTextFunction()
    {
        @Override
        protected String evaluate(String arg)
        {
            if (arg == null)
            {
                return null /* SSAS */;
            }

            return arg.trim();
        }
    };

    /**
     * An implementation of the MID function<br/>
     * MID returns a specific number of
     * characters from a text string, starting at the specified position.<p/>
     * <p/>
     * <b>Syntax<b>:<br/> <b>MID</b>(<b>text</b>, <b>start_num</b>,
     * <b>num_chars</b>)<br/>
     *
     * @author Manda Wilson &lt; wilson at c bio dot msk cc dot org &gt;
     */
    public static final OlapPluginUserJavaFunction MID = new OlapVBExcelTextFunctions(3, String.class, int.class, int.class)
    {
        @Override
        protected Object doEval(Object[] args) throws OlapPluginEvaluationException
        {
            String text = (String) args[0];

            if (text == null)
            {
                return null /* SSAS */;
            }

            Integer startCharNum = (Integer) args[1];
            Integer numChars = (Integer) args[2];
            int startIx = startCharNum - 1; // convert to zero-based

            // Note - for start_num arg, blank/zero causes error(#VALUE!),
            // but for num_chars causes empty string to be returned.
            if (startIx < 0)
            {
                throw new OlapPluginEvaluationException(OlapPluginEvaluationException.Kind.VALUE_INVALID);
            }
            if (numChars < 0)
            {
                throw new OlapPluginEvaluationException(OlapPluginEvaluationException.Kind.VALUE_INVALID);
            }
            int len = text.length();
            if (numChars < 0 || startIx > len)
            {
                throw new OlapPluginEvaluationException(OlapPluginEvaluationException.Kind.VALUE_INVALID);
            }
            int endIx = Math.min(startIx + numChars, len);
            String result = text.substring(startIx, endIx);
            return result;
        }
    };

    private static final class LeftRight extends OlapVBExcelTextFunctions
    {
        private final boolean _isLeft;

        protected LeftRight(boolean isLeft)
        {
            super(2, String.class, int.class);
            _isLeft = isLeft;
        }

        @Override
        protected Object doEval(Object[] args) throws OlapPluginEvaluationException
        {
            String arg = (String) args[0];

            if (arg == null)
            {
                return null /* SSAS */;
            }

            Integer index = (Integer) args[1];

            String result;
            if (_isLeft)
            {
                result = arg.substring(0, Math.min(arg.length(), index));
            }
            else
            {
                result = arg.substring(Math.max(0, arg.length() - index));
            }
            return result;
        }
    }

    public static final OlapVBExcelTextFunctions LEFT = new LeftRight(true);

    public static final OlapVBExcelTextFunctions RIGHT = new LeftRight(false);

    public static final OlapVBExcelTextFunctions CONCATENATE = new OlapVBExcelTextFunctions(2, 100, String.class)
    {
        @Override
        protected Object doEval(Object[] args) throws OlapPluginEvaluationException
        {
            StringBuffer sb = new StringBuffer();
            for (int i = 0, iSize = args.length; i < iSize; i++)
            {
                if (args[i] != null)
                {
                    sb.append(args[i]);
                }
            }
            return sb.toString();
        }

    };

    public static final OlapVBExcelTextFunctions EXACT = new OlapVBExcelTextFunctions(2, String.class, String.class)
    {
        @Override
        protected Object doEval(Object[] args) throws OlapPluginEvaluationException
        {
            String s0 = (String) args[0];
            String s1 = (String) args[1];
            return s0.equals(s1);
        }
    };

    public static final OlapVBExcelTextFunctions REPLACE = new OlapVBExcelTextFunctions(3, String.class, String.class, String.class)
    {
        @Override
        protected Object doEval(Object[] args) throws OlapPluginEvaluationException
        {
            String s0 = (String) args[0];
            String s1 = (String) args[1];
            String s2 = (String) args[2];
            return s0.replace(s1, s2);
        }
    };

    public static final OlapVBExcelTextFunctions INSTR = new OlapVBExcelTextFunctions(new Class[]{String.class, String.class}, new Class[]{Integer.class, String.class, String.class})
    {
        @Override
        protected Object doEval(Object[] args) throws OlapPluginEvaluationException
        {
            // ( s1, s2 )  or ( start, s1, s2 )

            if (args.length == 2)
            {
                String s0 = (String) args[0];

                if (s0 == null)
                {
                    return null /* SSAS */;
                }

                String s1 = (String) args[1];

                s0 = s0.toLowerCase();
                s1 = s1.toLowerCase();

                if (s1 == null)
                {
                    return s0.isEmpty() ? 0 : 1;
                }

                return 1 + s0.indexOf(s1);
            }
            else if (args.length == 3)
            {
                final int start = (Integer) args[0];

                String s0 = (String) args[1];

                if (s0 == null)
                {
                    return null /* SSAS */;
                }

                String s1 = (String) args[2];

                s0 = s0.toLowerCase();
                s1 = s1.toLowerCase();

                if (s1 == null)
                {
                    if (start <= s0.length())
                    {
                        return start;
                    }
                    else
                    {
                        return 0;
                    }
                }
                return 1 + s0.indexOf(s1, start - 1 /* VB is 1-based */);
            }

            throw new CdShouldNotBeHereProgrammingException();
        }
    };

}
