/*
 * Copyright 2014 - 2018 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.plugin.method;

import crazydev.common.javacompiler.DynamicClassLoader;
import crazydev.iccube.mdx.parser.ast.expression.function.MdxFunctionNativeBodyExpression;
import crazydev.iccube.olap.eval.execinstr.OlapPreparedInstr;
import crazydev.iccube.olap.eval.execinstr.gf.context.GFContext;
import crazydev.iccube.olap.eval.function.OlapExecutedWithPreparedInstr;
import crazydev.iccube.olap.eval.instr.OlapInstrLocationRange;
import crazydev.iccube.olap.eval.instr.OlapInstrVisitor;
import crazydev.iccube.olap.eval.method.OlapJavaMethod;
import crazydev.iccube.olap.eval.method.OlapJavaRegularMethod;
import crazydev.iccube.olap.eval.select.context.IOlapPrepareContext;
import crazydev.iccube.olap.eval.select.context.OlapEvaluationContext;
import crazydev.iccube.olap.loggers.OlapLoggers;
import javassist.bytecode.BadBytecode;
import javassist.bytecode.ClassFile;
import javassist.bytecode.MethodInfo;
import javassist.bytecode.SignatureAttribute;
import org.jetbrains.annotations.Nullable;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

public class OlapJModuleNativeFunctionInstr extends OlapExecutedWithPreparedInstr<OlapEvaluationContext>
{
    private final MdxFunctionNativeBodyExpression function;

    private final String sourceCode;

    private volatile OlapJavaMethod resolvedMethod;

    private ClassLoader resolvedMethodCL;

    public OlapJModuleNativeFunctionInstr(OlapInstrLocationRange range, MdxFunctionNativeBodyExpression function, String sourceCode)
    {
        super(range);

        this.function = function;
        this.sourceCode = sourceCode;
    }

    public MdxFunctionNativeBodyExpression getFunction()
    {
        return function;
    }

    public boolean isWithFunction()
    {
        return function.isWithFunction();
    }

    public String getFunctionName()
    {
        return function.getFunctionName();
    }

    public int getFunctionArgCount()
    {
        return function.getFunctionArgCount();
    }

    public String getFunctionArgName(int index)
    {
        return function.getFunctionArgName(index);
    }

    @Nullable
    public String getFunctionArgType(int index)
    {
        return function.getFunctionArgType(index);
    }

    public String getFunctionSourceCode()
    {
        return sourceCode;
    }

    @Override
    public boolean isNativeJava()
    {
        return true;
    }

    @Nullable
    @Override
    public MdxFunctionNativeBodyExpression getNativeFunctionSourceCode()
    {
        return function;
    }

    @Override
    public String typeForError()
    {
        return "JAVA NATIVE FUNCTION";
    }

    @Override
    public void accept(OlapInstrVisitor v)
    {
        if (v.visit(this))
        {
        }
    }

    @Override
    protected OlapPreparedInstr doPrepare(IOlapPrepareContext context)
    {
        return new OlapJModuleNativeFunctionPreparedInstr(this);
    }

    public OlapJavaMethod resolveJavaNativeMethod(GFContext context)
            throws ClassNotFoundException,
                   NoSuchMethodException
    {
        final ClassLoader cl = context.getNativeFunctionsClassLoader();

        if (resolvedMethod == null || resolvedMethodCL != cl /* ensure always latest version of method being used */)
        {
            return doResolveJavaNativeMethod(context);
        }

        return resolvedMethod;
    }

    private synchronized OlapJavaMethod doResolveJavaNativeMethod(GFContext context)
            throws ClassNotFoundException,
                   NoSuchMethodException
    {
        final ClassLoader cl = context.getNativeFunctionsClassLoader();

        if (resolvedMethodCL != cl /* ensure always latest version of method being used */)
        {
            resolvedMethodCL = cl;
            resolvedMethod = null;
        }

        if (resolvedMethod != null)
        {
            return resolvedMethod;
        }

        final String methodName = getFunctionName();
        final String className = OlapJavaMethod.NATIVE_JAVA_PACKAGE + "." + methodName;

        final Class<?> clazz = Class.forName(className, true, resolvedMethodCL);

        resolvedMethod = lookupMethod(clazz, methodName);

        final int offset = safeExtractOffset(cl, clazz, resolvedMethod);

        resolvedMethod.setOffset(offset);

        return resolvedMethod;
    }

    private static OlapJavaMethod lookupMethod(Class<?> clazz, String methodName) throws NoSuchMethodException
    {
        final Method[] methods = clazz.getMethods();

        for (Method method : methods)
        {
            if (method.getName().equals(methodName))
            {
                return new OlapJavaRegularMethod(method);
            }
        }

        throw new NoSuchMethodException(methodName);
    }

    private static int safeExtractOffset(ClassLoader cl, Class<?> clazz, OlapJavaMethod method)
    {
        if (cl instanceof DynamicClassLoader)
        {
            final byte[] bc = ((DynamicClassLoader) cl).getByteCode(clazz);

            if (bc != null)
            {
                return safeExtractOffset(bc, method);
            }
        }

        return -1;
    }

    private static int safeExtractOffset(byte[] byteCode, OlapJavaMethod method)
    {
        try
        {
            final DataInputStream in = new DataInputStream(new ByteArrayInputStream(byteCode));

            final ClassFile jClazz = new ClassFile(in);
            final MethodInfo jMethod = getMethodInfo(jClazz, method);

            if (jMethod != null)
            {
                return jMethod.getLineNumber(0);
            }

            return -1;

        }
        catch (Exception ex)
        {
            OlapLoggers.MDX_EVALUATION.warn("[java] could not extract offset for method [" + method.getName() + "]", ex);
            return -1;
        }

    }

    @Nullable
    private static MethodInfo getMethodInfo(ClassFile clazz, OlapJavaMethod method) throws BadBytecode
    {
        final List<MethodInfo> methodInfos = clazz.getMethods();

        if (methodInfos == null || methodInfos.isEmpty())
        {
            return null;
        }

        final List<MethodInfo> candidates = new ArrayList<>();

        for (MethodInfo methodInfo : methodInfos)
        {
            if (methodInfo.getName().equals(method.getName()))
            {
                candidates.add(methodInfo);
            }
        }

        if (candidates.isEmpty())
        {
            return null;
        }

        if (candidates.size() == 1)
        {
            return candidates.get(0);
        }

        // return the first w/ the same number of parameters (wait & see if more complex logic is required)

        final int pCount = method.getParamCount();

        for (MethodInfo candidate : candidates)
        {
            final String desc = candidate.getDescriptor();
            final SignatureAttribute.MethodSignature signature = SignatureAttribute.toMethodSignature(desc);

            final SignatureAttribute.Type[] types = signature.getParameterTypes();
            final int count = types != null ? types.length : 0;

            if (count == pCount)
            {
                return candidate;
            }
        }

        return null;
    }

    @Override
    public String toStringForDebugger()
    {
        return "JAVA NATIVE FUNCTION";
    }
}
