/*
 * Copyright 2014 - 2018 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.plugin.method;

import crazydev.iccube.exception.OlapErrorCode;
import crazydev.iccube.olap.entity.OlapEntity;
import crazydev.iccube.olap.entity.tuple.dimensionality.OlapTupleDimensionality;
import crazydev.iccube.olap.eval.exception.OlapEvaluationException;
import crazydev.iccube.olap.eval.execinstr.gf.context.GFContext;
import crazydev.iccube.olap.eval.execinstr.gf.nodes.GFNode;
import crazydev.iccube.olap.eval.execinstr.gf.tupleevaluator.GFTupleEvaluator;
import crazydev.iccube.olap.eval.instr.OlapInstrLocationRange;
import crazydev.iccube.olap.eval.method.OlapJavaMethod;
import crazydev.iccube.olap.eval.method.OlapJavaMethodHelper;
import crazydev.iccube.olap.loggers.OlapLoggers;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.lang.reflect.InvocationTargetException;

public class GFJNativeFunctionNode extends GFNode
{
    private final OlapJModuleNativeFunctionInstr instr;

    public GFJNativeFunctionNode(boolean markedForValueEvaluation, OlapJModuleNativeFunctionInstr instr)
    {
        super(markedForValueEvaluation);

        this.instr = instr;
    }

    @Override
    public OlapInstrLocationRange getLocationRange()
    {
        return instr.getLocationRange();
    }

    @Override
    public String getMdxForDrillthrough()
    {
        return instr.getMdxForDrillthrough();
    }

    @Nullable
    @Override
    public OlapTupleDimensionality getTupleDimensionality()
    {
        return null /* cannot be determined */;
    }

    @Override
    protected OlapEntity evaluate(GFContext context, GFTupleEvaluator evaluator)
    {
        if (!context.isJavaMdxNativesActivated())
        {
            throw new OlapEvaluationException(context, OlapErrorCode.NATIVE_METHOD_EXEC_ERROR, "functions", "", "JAVA native functions are not active");
        }

        if (instr.isWithFunction())
        {
            throw new OlapEvaluationException(context, OlapErrorCode.NATIVE_METHOD_EXEC_ERROR, "functions", "", "JAVA native functions cannot be defined in a SELECT statement (i.e., WITH NATIVE ...)");
        }

        OlapJavaMethod method = null;

        try
        {
            method = instr.resolveJavaNativeMethod(context);

            return doEvaluate(context, evaluator, method);

        }
        catch (ClassNotFoundException ex)
        {
            OlapLoggers.MDX_EVALUATION.error(getErrorMessage(method), ex);
            throw new OlapEvaluationException(context, OlapErrorCode.JAVA_CLASS_UNKNOWN, OlapJavaMethod.NATIVE_JAVA_PACKAGE + "." + instr.getFunctionName());
        }
        catch (InvocationTargetException ex)
        {
            OlapLoggers.MDX_EVALUATION.error(getErrorMessage(method), ex);
            throw new OlapEvaluationException(context, OlapErrorCode.METHOD_EXEC_ERROR, getModulePrefix(), getMethodName(), String.valueOf(ex.getTargetException()) /* getMessage might be null */);
        }
        catch (IllegalAccessException ex)
        {
            OlapLoggers.MDX_EVALUATION.error(getErrorMessage(method), ex);
            throw new OlapEvaluationException(context, OlapErrorCode.METHOD_EXEC_ERROR, getModulePrefix(), getMethodName(), String.valueOf(ex) /* getMessage might be null */);
        }
        catch (InstantiationException ex)
        {
            OlapLoggers.MDX_EVALUATION.error(getErrorMessage(method), ex);
            throw new OlapEvaluationException(context, OlapErrorCode.METHOD_FIELD_UNKNOWN, getModulePrefix(), getMethodName(), String.valueOf(ex) /* getMessage might be null */);
        }
        catch (NoSuchMethodException ex)
        {
            OlapLoggers.MDX_EVALUATION.error(getErrorMessage(method), ex);
            throw new OlapEvaluationException(context, OlapErrorCode.METHOD_UNKNOWN, getModulePrefix(), getMethodName(), String.valueOf(ex) /* getMessage might be null */);
        }
        catch (RuntimeException ex /* e.g., IllegalArgumentException */)
        {
            if (ex instanceof OlapEvaluationException)
            {
                throw ex;
            }

            OlapLoggers.MDX_EVALUATION.error(getErrorMessage(method), ex);
            throw new OlapEvaluationException(context, OlapErrorCode.METHOD_EXEC_ERROR, getModulePrefix(), getMethodName(), String.valueOf(ex) /* getMessage might be null */);
        }


    }

    @NotNull
    private String getErrorMessage(@Nullable OlapJavaMethod method)
    {
        if (method == null || method.getOffset() == -1)
        {
            return getModuleName() + " Method Error ( " + getMethodName() + " )";
        }

        return getModuleName() + " Method Error ( " + getMethodName() + " [code-offset:" + method.getOffset() + "] )";
    }

    private String getModuleName()
    {
        return "JAVA";
    }

    private String getModulePrefix()
    {
        return "JAVA!";
    }

    private String getMethodName()
    {
        return instr.getFunctionName();
    }

    private OlapEntity doEvaluate(GFContext context, GFTupleEvaluator evaluator, OlapJavaMethod method)
            throws IllegalAccessException,
                   InstantiationException,
                   InvocationTargetException
    {
        final Object javaResult;

        final int argCount = instr.getFunctionArgCount();

        if (argCount == 0)
        {
            javaResult = method.invoke(null /* static */);
        }
        else
        {
            final OlapEntity[] mdxParams = new OlapEntity[argCount];

            for (int ii = 0; ii < argCount; ii++)
            {
                mdxParams[ii] = context.lookupDeclaredFunctionArgument(ii);
            }

            final Object[] javaParams = OlapJavaMethodHelper.convertScalarEntityParamsToJavaNativeValues(getLocationRange(), method, mdxParams);

            javaResult = method.invoke(null /* static */, javaParams);
        }

        final OlapEntity mdxResult = OlapJavaMethodHelper.convertJavaResultToEntity(context, evaluator, javaResult);
        return mdxResult;
    }

}
