package crazydev.iccube.builder.googleapi.searchconsole.datasource;

import com.google.api.services.searchconsole.v1.SearchConsole;
import com.google.api.services.searchconsole.v1.model.ApiDataRow;
import com.google.api.services.searchconsole.v1.model.SearchAnalyticsQueryRequest;
import com.google.api.services.searchconsole.v1.model.SearchAnalyticsQueryResponse;
import crazydev.common.collection.CdCollections;
import crazydev.iccube.builder.OlapBuilderConnectionPool;
import crazydev.iccube.builder.OlapBuilderContext;
import crazydev.iccube.builder.datasource.reader.OlapBuilderAbstractTableRowReader;
import crazydev.iccube.builder.errors.OlapBuilderErrorCode;
import crazydev.iccube.builder.errors.OlapBuilderErrorException;
import crazydev.iccube.builder.googleapi.OlapBuilderGoogleCommon;
import crazydev.iccube.builder.model.def.IOlapBuilderDataColumnDef;
import crazydev.iccube.builder.model.def.IOlapBuilderTableRow;
import crazydev.iccube.builder.model.def.OlapBuilderAbstractTableRow;
import crazydev.iccube.olap.component.context.OlapRuntimeContext;
import org.jetbrains.annotations.Nullable;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class OlapBuilderGoogleSearchConsole4RowReader extends OlapBuilderAbstractTableRowReader<OlapBuilderGoogleSearchConsole4DataTable, OlapBuilderGoogleSearchConsoleConnection>
{
    private static final Map<Integer, DateTimeFormatter> formatters = new HashMap<>();

    static
    {
        formatters.put("yyyyMMdd".length(), DateTimeFormat.forPattern("yyyyMMdd"));
        formatters.put("yyyyMMddHH".length(), DateTimeFormat.forPattern("yyyyMMddHH"));
        formatters.put("yyyyMMddHHmm".length(), DateTimeFormat.forPattern("yyyyMMddHHmm"));
    }

    private List<ApiDataRow> rows;

    private int offset = 0;

    private final Map<IOlapBuilderDataColumnDef, Integer> colMapper = new HashMap<>();

    private int i = -1;

    private LocalDate startDate;

    protected OlapBuilderGoogleSearchConsole4RowReader(OlapBuilderContext context, OlapBuilderConnectionPool connectionPool, int maxRowCount, OlapBuilderGoogleSearchConsole4DataTable dataTable)
    {
        super(context, connectionPool, maxRowCount, dataTable, dataTable.getName());
    }

    @Override
    public void doInit()
    {
        initConnection();

        offset = 0;
        colMapper.clear();
        startDate = table.getTypedDS().getStartDate();
        startDate = startDate == null ? new LocalDate().minusMonths(1) : startDate;

        int idx = 0;
        for (final IOlapBuilderDataColumnDef column : this.table.getAllColumns())
        {
            if (column.getTableType() != null && column.getTableType().startsWith(OlapBuilderGoogleCommon.DIMENSION_PREFIX))
            {
                colMapper.put(column, idx++);
            }
        }

    }

    @Override
    public void done()
    {
        super.done();
    }

    @Override
    public boolean isRowSafe()
    {
        return true;
    }

    private void loadPage(OlapRuntimeContext context)
    {
        rows = Collections.EMPTY_LIST;

        final SearchAnalyticsQueryRequest request = table.buildRequest(startDate);
        if (request == null)
        {
            return;
        }
        // both start and end date are included
        startDate = OlapBuilderGoogleCommon.format.parseLocalDate(request.getEndDate()).plusDays(1);

        SearchAnalyticsQueryResponse response;
        try
        {
            final SearchConsole searchConsole = connection.getSearchConsole(context);
            response = searchConsole.searchanalytics()
                    .query(table.getTypedDS().getDomain(), request)
                    .execute();
            rows = response.getRows();
            rows = rows == null ? Collections.EMPTY_LIST : rows;
            i = 0;
        }
        catch (IOException e)
        {
            throw new OlapBuilderErrorException(OlapBuilderErrorCode.IO_ERROR, e);
        }
    }

    @Override
    public @Nullable IOlapBuilderTableRow doNextRow()
    {
        if (i == -1)
        {
            loadPage(context.asRuntimeContext());
        }

        if (i >= rows.size())
        {
            loadPage(context.asRuntimeContext());
        }

        if (rows.isEmpty())
        {
            return null;
        }
        final ApiDataRow row = rows.get(i++);
        return new OlapBuilderAbstractTableRow()
        {
            @Nullable
            @Override
            protected Object getDataImpl(IOlapBuilderDataColumnDef columnDef)
            {
                final String name = columnDef.getName();
                if (name == null)
                {
                    return null;
                }

                final Integer metricIdx = colMapper.get(columnDef);
                if (metricIdx != null)
                {
                    final List<String> keys = row.getKeys();
                    final String dimValue = CdCollections.getSafe(keys, metricIdx, null);
                    if (dimValue != null && name.endsWith("date"))
                    {
                        return safeParseDate(dimValue);
                    }
                    else
                    {
                        return dimValue;
                    }
                }
                if (name.equals(OlapBuilderGoogleCommon.METRIC_PREFIX + "CLICKS"))
                {
                    return row.getClicks();
                }
                if (name.equals(OlapBuilderGoogleCommon.METRIC_PREFIX + "POSITIONS"))
                {
                    return row.getPosition();
                }
                if (name.equals(OlapBuilderGoogleCommon.METRIC_PREFIX + "IMPRESSIONS"))
                {
                    return row.getImpressions();
                }
                if (name.equals(OlapBuilderGoogleCommon.METRIC_PREFIX + "CTR"))
                {
                    return row.getCtr();
                }

                return null;
            }

            private static Comparable<? extends Comparable<?>> safeParseDate(String dimValue)
            {
                try
                {
                    return LocalDate.parse(dimValue);
                }
                catch (Exception ex)
                {
                    return dimValue;
                }
            }

        };

    }
}