package crazydev.iccube.builder.googleapi.ga4.datasource;

import com.google.analytics.data.v1beta.*;
import crazydev.iccube.builder.OlapBuilderConnectionPool;
import crazydev.iccube.builder.OlapBuilderContext;
import crazydev.iccube.builder.datasource.reader.OlapBuilderAbstractTableRowReader;
import crazydev.iccube.builder.googleapi.OlapBuilderGoogleCommon;
import crazydev.iccube.builder.model.def.IOlapBuilderDataColumnDef;
import crazydev.iccube.builder.model.def.IOlapBuilderTableRow;
import crazydev.iccube.builder.model.def.OlapBuilderAbstractTableRow;
import crazydev.iccube.builder.type.OlapBuilderInputType;
import org.jetbrains.annotations.Nullable;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static crazydev.iccube.builder.googleapi.ga4.datasource.OlapBuilderGoogleAnalytics4DataTable.END_LOAD_DATE_COL_NAME;

public class OlapBuilderGoogleAnalytics4RowReader extends OlapBuilderAbstractTableRowReader<OlapBuilderGoogleAnalytics4DataTable, OlapBuilderGoogleAnalytics4Connection>
{
    private static final Map<Integer, DateTimeFormatter> formatters = new HashMap<>();

    static
    {
        formatters.put("yyyyMMdd".length(), DateTimeFormat.forPattern("yyyyMMdd"));
        formatters.put("yyyyMMddHH".length(), DateTimeFormat.forPattern("yyyyMMddHH"));
        formatters.put("yyyyMMddHHmm".length(), DateTimeFormat.forPattern("yyyyMMddHHmm"));
    }

    private final int pageSize;

    private List<Row> rows;

    private int offset = 0;

    private Map<String, Integer> colMapper = new HashMap<>();

    private int i = -1;

    private LocalDate endLoadTime;

    protected OlapBuilderGoogleAnalytics4RowReader(OlapBuilderContext context, OlapBuilderConnectionPool connectionPool, int maxRowCount, OlapBuilderGoogleAnalytics4DataTable olapBuilderGoogleAnalytics4DataTable)
    {
        super(context, connectionPool, maxRowCount, olapBuilderGoogleAnalytics4DataTable, "Google Analytics V4 Data Table");

        pageSize = this.table.getTypedDS().getPageSize();
    }

    @Override
    public void doInit()
    {
        initConnection();

        endLoadTime = null;
        offset = 0;
        colMapper.clear();

        loadPage();
    }

    @Override
    public void done()
    {
        super.done();
    }

    @Override
    public boolean isRowSafe()
    {
        return true;
    }

    private void loadPage()
    {
        final RunReportRequest request = table.buildRequest(context, offset);
        if (request.getDateRangesCount() == 1)
        {
            final String endDateS = request.getDateRanges(0).getEndDate();
            endLoadTime = OlapBuilderGoogleCommon.format.parseLocalDate(endDateS);
        }

        final RunReportResponse response = connection.getAnalyticsData(context.getRequestContext().asRuntimeContext()).runReport(request);
        rows = response.getRowsList();
        offset += rows.size();

        if (colMapper.isEmpty())
        {
            final List<DimensionHeader> listD = response.getDimensionHeadersList();
            for (int i = 0; i < listD.size(); i++)
            {
                colMapper.put(OlapBuilderGoogleCommon.DIMENSION_PREFIX + listD.get(i).getName(), i);
            }
            final List<MetricHeader> listM = response.getMetricHeadersList();
            for (int i = 0; i < listM.size(); i++)
            {
                colMapper.put(OlapBuilderGoogleCommon.METRIC_PREFIX + listM.get(i).getName(), i);
            }
        }
        i = 0;
    }

    @Override
    public @Nullable IOlapBuilderTableRow doNextRow()
    {
        if (i >= rows.size())
        {
            // is the last page ?
            if (rows.size() < pageSize)
            {
                return null;
            }
            //no , load the next page
            loadPage();
        }
        if (i >= rows.size())
        {
            return null;
        }

        final Row row = rows.get(i++);
        return new OlapBuilderAbstractTableRow()
        {
            @Nullable
            @Override
            protected Object getDataImpl(IOlapBuilderDataColumnDef columnDef)
            {
                if (columnDef.getName().equals(END_LOAD_DATE_COL_NAME))
                {
                    return endLoadTime;
                }

                final String name = columnDef.getTableType();
                final Integer index = colMapper.get(name);
                if (index == null || name == null)
                {
                    return null;
                }
                if (name.startsWith(OlapBuilderGoogleCommon.DIMENSION_PREFIX))
                {
                    final DimensionValue val = row.getDimensionValues(index);
                    return toTyped(columnDef.getType(), val.getValue());
                }
                if (name.startsWith(OlapBuilderGoogleCommon.METRIC_PREFIX))
                {
                    final MetricValue val = row.getMetricValues(index);
                    return val.getValue();
                }
                return null;
            }

            private Comparable toTyped(OlapBuilderInputType type, String value)
            {
                DateTimeFormatter format;
                switch (type)
                {
                    case DATE ->
                    {
                        try
                        {
                            format = formatters.get(value.length());
                            return format == null ? value : format.parseLocalDate(value);
                        }
                        catch (Exception ex)
                        {
                            return value;
                        }
                    }
                    case DATETIME ->
                    {
                        try
                        {
                            format = formatters.get(value.length());
                            return format == null ? value : format.parseLocalDateTime(value);
                        }
                        catch (Exception ex)
                        {
                            return value;
                        }
                    }
                    default ->
                    {
                        return value;
                    }
                }
            }
        };

    }

}
