/*
 * Copyright 2014 - 2019 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.googleapi.ga4.datasource;

import com.google.analytics.data.v1beta.*;
import crazydev.common.property.CdProperty;
import crazydev.common.property.CdReadWriteProperty;
import crazydev.common.xml.CdLocalDateXmlAdapter;
import crazydev.iccube.builder.OlapBuilderConnectionPool;
import crazydev.iccube.builder.OlapBuilderContext;
import crazydev.iccube.builder.datasource.reader.IOlapBuilderTableRowReader;
import crazydev.iccube.builder.errors.OlapBuilderErrorManager;
import crazydev.iccube.builder.googleapi.OlapBuilderGoogleCommon;
import crazydev.iccube.builder.model.def.IOlapBuilderDataColumnDef;
import crazydev.iccube.builder.model.def.IOlapBuilderDataTableDef;
import crazydev.iccube.builder.model.impl.OlapBuilderDataColumn;
import crazydev.iccube.builder.model.impl.table.OlapBuilderBaseDataTable;
import crazydev.iccube.builder.type.OlapBuilderInputType;
import crazydev.iccube.olap.component.context.OlapRuntimeContext;
import jakarta.xml.bind.annotation.XmlAccessType;
import jakarta.xml.bind.annotation.XmlAccessorType;
import jakarta.xml.bind.annotation.XmlAttribute;
import jakarta.xml.bind.annotation.XmlRootElement;
import jakarta.xml.bind.annotation.adapters.XmlJavaTypeAdapter;
import org.jetbrains.annotations.Nullable;
import org.joda.time.LocalDate;

import java.util.*;
import java.util.stream.Collectors;

import static crazydev.iccube.builder.googleapi.OlapBuilderGoogleCommon.format;

@XmlRootElement(name = "GA4DataTable")
@XmlAccessorType(XmlAccessType.FIELD)
public class OlapBuilderGoogleAnalytics4DataTable extends OlapBuilderBaseDataTable<OlapBuilderGoogleAnalytics4Connection>
{
    public static final CdProperty START_DATE = new CdReadWriteProperty(OlapBuilderGoogleAnalytics4DataTable.class, "startDate", false);

    public static final CdProperty END_DATE = new CdReadWriteProperty(OlapBuilderGoogleAnalytics4DataTable.class, "endDate", false);

    public static final CdProperty DIMENSIONS = new CdReadWriteProperty(OlapBuilderGoogleAnalytics4DataTable.class, "dimensions");

    public static final CdProperty METRICS = new CdReadWriteProperty(OlapBuilderGoogleAnalytics4DataTable.class, "metrics");

    public static final String END_LOAD_DATE_COL_NAME = "icEndLoadDate";

    @XmlAttribute(required = true)
    protected String metrics;

    @XmlAttribute(required = true)
    protected String dimensions;

    @Nullable
    @XmlJavaTypeAdapter(value = CdLocalDateXmlAdapter.class)
    @XmlAttribute(required = false)
    private LocalDate startDate;

    @Nullable
    @XmlJavaTypeAdapter(value = CdLocalDateXmlAdapter.class)
    @XmlAttribute(required = false)
    private LocalDate endDate;

    protected OlapBuilderGoogleAnalytics4DataTable()
    {
    }

    protected OlapBuilderGoogleAnalytics4DataTable(String tableName)
    {
        super(tableName);
    }

    OlapBuilderGoogleAnalytics4DataSource getTypedDS()
    {
        return (OlapBuilderGoogleAnalytics4DataSource) getDataSource();
    }

    @Override
    public boolean isNameReadOnly()
    {
        return false;
    }

    @Override
    public String getInternalFriendlyTypeName()
    {
        return "GA4 Table";
    }

    @Override
    public boolean isDiscoveredInDataSource()
    {
        return false;
    }

    @Override
    protected List<? extends IOlapBuilderDataColumnDef> doDiscoverAllColumns(OlapRuntimeContext context, OlapBuilderGoogleAnalytics4Connection connection, OlapBuilderErrorManager errorManager)
    {
        return doDiscoverAllColumnsForRefresh(context, connection, errorManager);
    }

    protected List<? extends IOlapBuilderDataColumnDef> doDiscoverAllColumnsForRefresh(OlapRuntimeContext context, OlapBuilderGoogleAnalytics4Connection connection, OlapBuilderErrorManager errorManager)
    {
        // =========================================================================================================
        // I'm returning all the current columns => the table won't be changed.
        // Remember : return a new list (the caller is assuming a #discover() )
        // =========================================================================================================

        final List<IOlapBuilderDataColumnDef> fake = new ArrayList<IOlapBuilderDataColumnDef>();
        if (dimensions != null)
        {
            Arrays.stream(dimensions.split(",")).map(String::trim).forEach(s -> {
                fake.add(new OlapBuilderDataColumn(OlapBuilderInputType.STRING, OlapBuilderGoogleCommon.DIMENSION_PREFIX + s, "fake"));
            });
        }
        if (metrics != null)
        {
            Arrays.stream(metrics.split(",")).map(String::trim).forEach(s -> {
                fake.add(new OlapBuilderDataColumn(OlapBuilderInputType.DOUBLE, OlapBuilderGoogleCommon.METRIC_PREFIX + s, "fake"));
            });
        }

        // map actual names for the UI to the name of the column and the type to the api name
        final Metadata result = connection.getAnalyticsData(context).getMetadata(getMetatDataRequest());
        final List<OlapBuilderDataColumn> decoratedColumnns = fake.stream().map(col -> {
            final String name = col.getTableType();
            if (name.startsWith(OlapBuilderGoogleCommon.METRIC_PREFIX))
            {
                final MetricMetadata meta = searchMetric(result.getMetricsList(), name.substring(2));
                if (meta != null)
                {
                    return new OlapBuilderDataColumn(OlapBuilderInputType.STRING, OlapBuilderGoogleCommon.METRIC_PREFIX + meta.getApiName(), meta.getUiName());
                }
                return null;
            }
            else
            {
                final DimensionMetadata meta = searchDimension(result.getDimensionsList(), name.substring(2));
                if (meta != null)
                {
                    OlapBuilderInputType type = OlapBuilderInputType.STRING;
                    if (meta.getCategory().equals("Time"))
                    {
                        type = meta.getDescription().contains("YYYYMMDDHH") ? OlapBuilderInputType.DATETIME : OlapBuilderInputType.DATE;
                    }
                    return new OlapBuilderDataColumn(type, OlapBuilderGoogleCommon.DIMENSION_PREFIX + meta.getApiName(), meta.getUiName());
                }
                return null;

            }
        }).filter(Objects::nonNull).collect(Collectors.toList());

        decoratedColumnns.add(new OlapBuilderDataColumn(OlapBuilderInputType.DATE, "iCube", END_LOAD_DATE_COL_NAME));

        return decoratedColumnns;
    }

    private GetMetadataRequest getMetatDataRequest()
    {
        return GetMetadataRequest.newBuilder().setName("properties/" + getTypedDS().getPropertyId() + "/metadata").build();
    }

    private static MetricMetadata searchMetric(List<MetricMetadata> metricCompatibilitiesList, String search)
    {
        final Optional<MetricMetadata> find = metricCompatibilitiesList.stream().filter(mess -> search.equals(mess.getApiName())).findFirst();
        return find.orElse(null);
    }

    private static DimensionMetadata searchDimension(List<DimensionMetadata> metricCompatibilitiesList, String search)
    {
        final Optional<DimensionMetadata> find = metricCompatibilitiesList.stream().filter(mess -> search.equals(mess.getApiName())).findFirst();
        return find.orElse(null);
    }

    RunReportRequest buildRequest(OlapBuilderContext context, int offset)
    {
        final RunReportRequest.Builder builder = RunReportRequest.newBuilder()
                .setLimit(getTypedDS().getPageSize())
                .setOffset(offset)
                .setProperty("properties/" + getTypedDS().getPropertyId())
                .addAllDimensions(getDimensions(getAllColumns()))
                .addAllMetrics(getMetrics(getAllColumns()))
                .addDateRanges(getDataRange());

        // do we are in an incremental load context
        final Comparable incrLoadMarker = context.getIncrementalLoadMarker(this);
        if (incrLoadMarker != null)
        {
            final String startDate1 = ((LocalDate) incrLoadMarker).toString(format);
            final String endDate1 = builder.getDateRanges(0).getEndDate();
            builder.setDateRanges(0, DateRange.newBuilder().setStartDate(startDate1).setEndDate(endDate1).build());
        }
        return builder.build();
    }

    private Iterable<Metric> getMetrics(List<IOlapBuilderDataColumnDef> cols)
    {
        return cols.stream()
                .filter(col -> col.getTableType().startsWith(OlapBuilderGoogleCommon.METRIC_PREFIX))
                .map(col -> Metric.newBuilder().setName(col.getTableType().substring(2)).build())
                .toList();
    }

    private Iterable<Dimension> getDimensions(List<IOlapBuilderDataColumnDef> cols)
    {
        return cols.stream()
                .filter(col -> col.getTableType().startsWith(OlapBuilderGoogleCommon.DIMENSION_PREFIX))
                .map(col -> Dimension.newBuilder().setName(col.getTableType().substring(2)).build())
                .toList();
    }

    private DateRange.Builder getDataRange()
    {
        final LocalDate startDate = this.startDate == null ? getTypedDS().getStartDate() : this.startDate;
        final LocalDate endDate = this.endDate == null ? getTypedDS().getEndDate() : this.endDate;

        DateRange.Builder builder = DateRange.newBuilder();
        if (startDate != null)
        {
            builder = builder.setStartDate(startDate.toString(format));
        }
        else
        {
            builder = builder.setStartDate(new LocalDate(2015, 8, 14).toString(format));
        }
        if (endDate != null)
        {
            builder = builder.setEndDate(endDate.toString(format));
        }
        else
        {
            builder = builder.setEndDate(LocalDate.now().minusDays(1).toString(format));
        }
        return builder;
    }

    @Override
    protected IOlapBuilderTableRowReader<OlapBuilderGoogleAnalytics4Connection> doCreateFullTableRowReader(OlapBuilderContext context, OlapBuilderConnectionPool connectionPool, int maxRowCount)
    {
        return new OlapBuilderGoogleAnalytics4RowReader(context, connectionPool, maxRowCount, this);
    }

    @Override
    public boolean isRefreshColumnOnUpdate(IOlapBuilderDataTableDef newTable)
    {
        if (super.isRefreshColumnOnUpdate(newTable))
        {
            return true;
        }

        final OlapBuilderGoogleAnalytics4DataTable tableUpdate = (OlapBuilderGoogleAnalytics4DataTable) newTable;

        return !Objects.equals(this.dimensions, tableUpdate.dimensions)
               || !Objects.equals(this.metrics, tableUpdate.metrics);
    }

}
