/*
 * Copyright 2014 - 2019 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.googleapi.drive.datasource;

import com.google.api.client.auth.oauth2.TokenResponseException;
import com.google.api.client.googleapis.json.GoogleJsonResponseException;
import com.google.api.services.drive.Drive;
import com.google.api.services.drive.model.File;
import crazydev.common.collection.CdFilter;
import crazydev.iccube.builder.errors.OlapBuilderErrorException;
import crazydev.iccube.builder.excel.datasource.OlapBuilderExcelConnection;
import crazydev.iccube.builder.googleapi.errors.OlapGoogleApiError;
import crazydev.iccube.olap.component.context.OlapRuntimeContext;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;

public class OlapBuilderGoogleSpreadsheetConnection extends OlapBuilderGoogleDriveConnection<OlapBuilderGoogleSpreadsheetDataSource>
{
    public static final String EXCEL_MIME_TYPE = "application/vnd.ms-excel";

    private final boolean reuse;

    private String fileId;

    private String fileName;

    private OlapBuilderExcelConnection excelConn;

    public OlapBuilderGoogleSpreadsheetConnection(OlapBuilderGoogleSpreadsheetDataSource dataSource, boolean reuse)
    {
        super(dataSource);
        this.reuse = reuse;
        fileId = dataSource.getFileId();
        fileName = dataSource.getFileName();

    }

    @Override
    public void onOpen(OlapRuntimeContext context) throws OlapBuilderErrorException
    {
        super.onOpen(context);

        try
        {
            final File file = getFile(jsonClient, fileId, fileName);
            final InputStream content;
            if (file.getMimeType().equals(EXCEL_MIME_TYPE))
            {
                final Drive.Files.Get exportReq = jsonClient.files().get(file.getId());
                content = exportReq.executeMediaAsInputStream();
            }
            else
            {
                final Drive.Files.Export exportReq = jsonClient.files().export(file.getId(), "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
                content = exportReq.executeMediaAsInputStream();
            }

            excelConn = OlapBuilderExcelConnection.openFromInputStream(content, "gdrive:" + fileId, dataSource.isWithoutFormulas());
        }
        catch (GoogleJsonResponseException ex)
        {
            throw OlapGoogleApiError.error("Table", fileName, ex);
        }
        catch (TokenResponseException ex)
        {
            throw OlapGoogleApiError.error("Table", fileName, ex);
        }
        catch (IOException ex)
        {
            throw new RuntimeException(ex);
        }
    }

    @Override
    protected void onClose()
    {
        super.onClose();

        // keep this open to allow caching
        if (reuse)
        {
            forceIsOpen();
        }
        else
        {
            if (excelConn != null)
            {
                excelConn.close();
            }
            excelConn = null;
        }
    }

    public OlapBuilderExcelConnection getExcelConn()
    {
        return excelConn;
    }

    public List<String> getSheetNames(CdFilter<String> filter)
    {
        return excelConn.getSheetNames(filter);
    }

    @Override
    protected String getMimeQ()
    {
        return " and (mimeType = '" + EXCEL_MIME_TYPE + "' OR mimeType contains 'spreadsheet')";
    }
}
