/*
 * Copyright 2014 - 2019 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.googleapi.drive.datasource;

import com.google.api.client.http.HttpRequestInitializer;
import com.google.api.client.http.HttpTransport;
import com.google.api.client.json.JsonFactory;
import com.google.api.services.drive.Drive;
import com.google.api.services.drive.DriveScopes;
import com.google.api.services.drive.model.File;
import com.google.api.services.drive.model.FileList;
import crazydev.common.utils.CdStringUtils;
import crazydev.iccube.builder.errors.OlapBuilderErrorException;
import crazydev.iccube.builder.googleapi.common.OlapBuilderGoogleApiConnection;
import crazydev.iccube.builder.googleapi.errors.OlapGoogleApiError;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

public abstract class OlapBuilderGoogleDriveConnection<DS extends OlapBuilderGoogleDriveDataSource> extends OlapBuilderGoogleApiConnection<Drive, DS>
{
    public OlapBuilderGoogleDriveConnection(DS dataSource)
    {
        super(dataSource);
    }

    @Override
    protected Drive buildClient(HttpRequestInitializer initializer, HttpTransport httpTransport, JsonFactory jsonFactory)
    {
        return new Drive.Builder(httpTransport, jsonFactory, initializer).setApplicationName(APPLICATION_NAME).build();
    }

    public List<String> getFileNames(@Nullable String filter) throws IOException
    {
        Drive.Files.List request = jsonClient.files().list();
        if (CdStringUtils.isNotNullAndNotBlank(filter))
        {
            request.setQ("name= '" + googleEscape(filter) + "'");
        }

        final List<String> names = new ArrayList<>();
        do
        {
            final FileList files = request.execute();
            files.getFiles().forEach(file -> names.add(file.getName()));
            request.setPageToken(files.getNextPageToken());
        } while (names.size() < 100 && request.getPageToken() != null && request.getPageToken().length() > 0);

        return names;
    }

    protected String getMimeQ()
    {
        return "";
    }

    File getFile(Drive jsonClient, @Nullable String fileId, String fileName) throws IOException
    {
        if (CdStringUtils.isNotNullAndNotBlank(fileId))
        {
            return jsonClient.files().get(fileId).execute();
        }
        else
        {
//            https://developers.google.com/drive/api/v3/search-files
            Drive.Files.List request = jsonClient.files().list();
            request.setQ("name= '" + googleEscape(fileName) + "'" + getMimeQ());

            do
            {
                FileList files = request.execute();

                for (File file : files.getFiles())
                {
                    final String filename = file.getName();
                    if (filename.equals(fileName))
                    {
                        return file;
                    }

                }
                request.setPageToken(files.getNextPageToken());
            } while (request.getPageToken() != null && request.getPageToken().length() > 0);

            throw new OlapBuilderErrorException(OlapGoogleApiError.GOOGLE_DRIVE_MISSING_FILE_NAME, fileName);
        }
    }

    private static String googleEscape(String fileName)
    {
        return fileName.replaceAll("'", "\\'");
    }

    @Override
    public String getAccountScope()
    {
        return DriveScopes.DRIVE_READONLY;
    }

    static void checkCredentialAndThrow(HttpRequestInitializer credential)
    {
        try
        {
            final Drive client = buildClient(credential);
            client.drives().list().execute();
        }
        catch (IOException e)
        {
            throw new RuntimeException("No Authorization to access Google Drive : " + e.getMessage());
        }
    }

    File checkFile(HttpRequestInitializer credential, String fileId, String fileName) throws IOException
    {
        return getFile(buildClient(credential), fileId, fileName);
    }

    @NotNull
    private static Drive buildClient(HttpRequestInitializer credential)
    {
        return new Drive.Builder(HTTP_TRANSPORT, JSON_FACTORY, credential).setApplicationName(APPLICATION_NAME).build();
    }

}
