/*
 * Copyright 2014 - 2020 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.googleapi.bigquery.datasource;

import com.google.cloud.bigquery.*;
import crazydev.common.utils.CdStringUtils;
import crazydev.iccube.builder.OlapBuilderConnectionPool;
import crazydev.iccube.builder.OlapBuilderContext;
import crazydev.iccube.builder.datasource.reader.IOlapBuilderTableRowReader;
import crazydev.iccube.builder.errors.OlapBuilderErrorManager;
import crazydev.iccube.builder.model.def.IOlapBuilderDataColumnDef;
import crazydev.iccube.builder.model.impl.OlapBuilderDataColumn;
import crazydev.iccube.builder.model.impl.table.OlapBuilderBaseDataTable;
import crazydev.iccube.builder.type.OlapBuilderInputType;
import crazydev.iccube.olap.component.context.OlapRuntimeContext;
import crazydev.iccube.olap.loggers.OlapLoggers;
import org.jetbrains.annotations.Nullable;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.DateTimeFormatterBuilder;
import org.joda.time.format.ISODateTimeFormat;

import java.io.IOException;
import java.time.LocalTime;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

public abstract class OlapBuilderGoogleBigqueryBaseDataTable extends OlapBuilderBaseDataTable<OlapBuilderGoogleBigqueryConnection>
{
    protected OlapBuilderGoogleBigqueryBaseDataTable()
    {
    }

    public OlapBuilderGoogleBigqueryBaseDataTable(String tableName)
    {
        super(tableName);
    }

    @Override
    protected IOlapBuilderTableRowReader<OlapBuilderGoogleBigqueryConnection> doCreateFullTableRowReader(OlapBuilderContext context, OlapBuilderConnectionPool connectionPool, int maxRowCount)
    {
        return new OlapBuilderGoogleBigQueryDataTableRowReader(context, connectionPool, maxRowCount, this);
    }

    @Override
    protected List<? extends IOlapBuilderDataColumnDef> doDiscoverAllColumnsForRefresh(OlapRuntimeContext context, OlapBuilderGoogleBigqueryConnection connection, OlapBuilderErrorManager errorManager)
    {
        return discoverAllColumns(context, connection, errorManager);
    }

    @Override
    protected List<? extends IOlapBuilderDataColumnDef> doDiscoverAllColumns(OlapRuntimeContext context, OlapBuilderGoogleBigqueryConnection openedConnection, OlapBuilderErrorManager errorManager)
    {
        try
        {
            final Schema schema = executeQuery4Discover(openedConnection, this);
            return toColumns(schema.getFields());
        }
        catch (IOException | InterruptedException ex)
        {
            throw new RuntimeException(ex);
        }

    }

    protected TableResult executeQuery(OlapBuilderGoogleBigqueryConnection connection, int maxRowCount, OlapBuilderGoogleBigqueryBaseDataTable table, @Nullable Comparable incrLoadMarker) throws IOException, InterruptedException
    {
        final Job job = createJob(connection, maxRowCount, table, incrLoadMarker);

        return maxRowCount > 0
               ? job.getQueryResults(BigQuery.QueryResultsOption.pageSize(maxRowCount))
               : job.getQueryResults()
                ;
    }

    protected Schema executeQuery4Discover(OlapBuilderGoogleBigqueryConnection connection, OlapBuilderGoogleBigqueryBaseDataTable table) throws IOException, InterruptedException
    {
        // this is for free
        Job queryJob = createJob(connection, 0, table, null);
        JobStatistics.QueryStatistics statistics = queryJob.getStatistics();
        final Schema schema = statistics.getSchema();
        if (schema != null)
        {
            return schema;
        }
        // bad luck we need to make a query
        final TableResult result = executeQuery(connection, 1, table, null);
        return result.getSchema();
    }

    private Job createJob(OlapBuilderGoogleBigqueryConnection connection, int maxRowCount, OlapBuilderGoogleBigqueryBaseDataTable table, @Nullable Comparable incrLoadMarker) throws InterruptedException
    {
        final String query = table.buildSelect(connection, incrLoadMarker);

        OlapLoggers.BUILDER_GOOGLE.debug(String.format(
                "[big-query] table [%s] select statement: %s", getName(), query
        ));

        final QueryJobConfiguration.Builder builder = QueryJobConfiguration.newBuilder(query)
                .setUseLegacySql(useLegacyCode());

        // settings
        OlapBuilderGoogleBigqueryDataSource ds = (OlapBuilderGoogleBigqueryDataSource) getDataSource();
        if (CdStringUtils.isNotNullAndNotBlank(ds.getDatasetId()))
        {
            builder.setDefaultDataset(ds.getDatasetId());
        }
        if (ds.getWaitTime() != null && ds.getWaitTime() != 0)
        {
            builder.setJobTimeoutMs(1000 * (long) ds.getWaitTime());
        }

        // discover
        final boolean dryRun = maxRowCount == 0;
        if (dryRun)
        {
            builder.setDryRun(true);
        }

        QueryJobConfiguration queryConfig = builder.build();

// Create a job ID so that we can safely retry.
        JobId jobId = JobId.of(UUID.randomUUID().toString());
        final JobInfo jobInfo = JobInfo.newBuilder(queryConfig).setJobId(jobId).build();
        Job queryJob = connection.getJsonClient().create(jobInfo);
        if (!dryRun)
        {
            // Wait for the query to complete.
            queryJob = queryJob.waitFor();
        }
        // Check for errors
        if (queryJob == null)
        {
            throw new RuntimeException("Job no longer exists");
        }
        else if (queryJob.getStatus().getError() != null)
        {
            // You can also look at queryJob.getStatus().getExecutionErrors() for all
            // errors, not just the latest one.
            throw new RuntimeException(queryJob.getStatus().getError().toString());
        }
        return queryJob;
    }

    protected abstract boolean useLegacyCode();

    protected String getProjectId(OlapBuilderGoogleBigqueryConnection connection)
    {
        return connection.getDataSource().getProjectId();
    }

    protected String getDataSetId(OlapBuilderGoogleBigqueryConnection connection)
    {
        return connection.getDataSource().getDatasetId();
    }

    protected abstract String buildSelect(OlapBuilderGoogleBigqueryConnection connection, @Nullable Comparable incrLoadMarker);

    protected List<? extends IOlapBuilderDataColumnDef> toColumns(FieldList fields)
    {
        final List<OlapBuilderDataColumn> cols = new ArrayList<OlapBuilderDataColumn>();

        for (Field field : fields)
        {
            cols.add(toColumn(field));
        }
        return cols;
    }

    private OlapBuilderDataColumn toColumn(Field field)
    {
        final StandardSQLTypeName type = field.getType().getStandardType();
        return new OlapBuilderDataColumn(toColumnType(type), type.toString(), field.getName());
    }

    private OlapBuilderInputType toColumnType(StandardSQLTypeName type)
    {
        switch (type)
        {
            case GEOGRAPHY:
            case STRING:
                return OlapBuilderInputType.STRING;
            case INT64:
                return OlapBuilderInputType.LONG;
            case NUMERIC:
            case FLOAT64:
                return OlapBuilderInputType.DOUBLE;
            case BOOL:
                return OlapBuilderInputType.BOOLEAN;
            case DATETIME:
            case TIMESTAMP:
                return OlapBuilderInputType.DATETIME;
            case DATE:
                return OlapBuilderInputType.DATE;
            case TIME:
                return OlapBuilderInputType.LONG;
            case BYTES:
            case STRUCT:
            case ARRAY:
            default:
                return OlapBuilderInputType.JAVA_OBJECT;
        }
    }

    @Nullable
    public Integer getWaitTime()
    {
        return ((OlapBuilderGoogleBigqueryDataSource) getDataSource()).getWaitTime();
    }

    protected String setupIncrLoadRestriction(String tableNameQ, Comparable incrLoadMarker)
    {
        final String columnName = getIncrementalLoadColumnName();
        final IOlapBuilderDataColumnDef columnDef = getSelectedColumn(columnName);
        final String columnTableType = columnDef.getTableType();

        final String restriction = columnName + " > " + setupIncrLoadRestrictionValue(tableNameQ, incrLoadMarker);

        OlapLoggers.BUILDER_GOOGLE.debug(String.format(
                "[big-query] table [%s] setup incr. load restriction [%s] [marker:%s] [marker-class:%s] [col-type:%s]",
                tableNameQ, restriction, incrLoadMarker, incrLoadMarker.getClass().getName(), columnTableType
        ));

        return restriction;
    }

    protected String setupIncrLoadRestrictionValue(String tableNameQ, Comparable incrLoadMarker)
    {
        final String columnName = getIncrementalLoadColumnName();
        final IOlapBuilderDataColumnDef columnDef = getSelectedColumn(columnName);
        final String columnTableType = columnDef.getTableType();

        String restriction = "" + incrLoadMarker;

        if ((incrLoadMarker instanceof Number) && "TIME".equals(columnTableType))
        {
            final LocalTime time = LocalTime.ofNanoOfDay(1000 * ((Number) incrLoadMarker).longValue());

            restriction = "TIME(DATETIME \"2020-01-01 " + time + "\")";
        }
        else if ((incrLoadMarker instanceof LocalDateTime) && "TIMESTAMP".equals(columnTableType))
        {
            // Does not work properly => toDate() is adding a timezone discrepancy
            // final long epoch = ((LocalDateTime) marker).toDate().getTime();

            final DateTimeFormatter dateTimeF = new DateTimeFormatterBuilder()
                    .append(ISODateTimeFormat.date())
                    .appendLiteral(" ")
                    .append(ISODateTimeFormat.time())
                    .toFormatter();

            final String dateTime = dateTimeF.print((LocalDateTime) incrLoadMarker);

            restriction = "TIMESTAMP(\"" + dateTime + "\")";
        }
        else if ((incrLoadMarker instanceof LocalDateTime) && "DATETIME".equals(columnTableType))
        {
            // Does not work properly => toDate() is adding a timezone discrepancy
            // final long epoch = ((LocalDateTime) marker).toDate().getTime();

            final DateTimeFormatter dateTimeF = new DateTimeFormatterBuilder()
                    .append(ISODateTimeFormat.date())
                    .appendLiteral(" ")
                    .append(ISODateTimeFormat.time())
                    .toFormatter();

            final String dateTime = dateTimeF.print((LocalDateTime) incrLoadMarker);

            restriction = "DATETIME(TIMESTAMP \"" + dateTime + "\")";
        }
        else if ((incrLoadMarker instanceof LocalDate) && "DATE".equals(columnTableType))
        {
            final LocalDate date = (LocalDate) incrLoadMarker;

            restriction = "DATE(" + date.getYear() + "," + date.getMonthOfYear() + "," + date.getDayOfMonth() + ")";
        }

        OlapLoggers.BUILDER_GOOGLE.debug(String.format(
                "[big-query] table [%s] setup incr. load restriction value [%s] [marker:%s] [marker-class:%s] [col-type:%s]",
                tableNameQ, restriction, incrLoadMarker, incrLoadMarker.getClass().getName(), columnTableType
        ));

        return restriction;
    }

}
