/*
 * Copyright 2014 - 2020 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.googleapi.bigquery.datasource;

import com.google.api.gax.rpc.ServerStream;
import com.google.api.gax.rpc.ServerStreamingCallable;
import com.google.auth.Credentials;
import com.google.cloud.bigquery.TableId;
import com.google.cloud.bigquery.storage.v1.*;
import crazydev.iccube.builder.OlapBuilderConnectionPool;
import crazydev.iccube.builder.OlapBuilderContext;
import crazydev.iccube.builder.datasource.reader.OlapBuilderAbstractTableRowReader;
import crazydev.iccube.builder.errors.OlapBuilderErrorException;
import crazydev.iccube.builder.googleapi.errors.OlapGoogleApiError;
import crazydev.iccube.builder.model.def.IOlapBuilderDataColumnDef;
import crazydev.iccube.builder.model.def.IOlapBuilderTableRow;
import crazydev.iccube.olap.loggers.OlapLoggers;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.util.List;

public class OlapBuilderGoogleBigQueryStorageDataTableRowReader
        extends OlapBuilderAbstractTableRowReader<OlapBuilderGoogleBigqueryDataTable, OlapBuilderGoogleBigqueryConnection>
{
    @Nullable
    private String tableNameQ;

    @Nullable
    private BigQueryReadClient client;

    @Nullable
    private OlapBuilderGoogleBigQueryStorageStreams streams;

    private int rowsCount;

    public OlapBuilderGoogleBigQueryStorageDataTableRowReader(OlapBuilderContext context, OlapBuilderConnectionPool connectionPool, int maxRowCount, OlapBuilderGoogleBigqueryDataTable table)
    {
        super(context, connectionPool, maxRowCount, table, table.getName());
    }

    @Override
    public void doInit()
    {
        initConnection();

        final OlapBuilderGoogleBigqueryDataSource ds = (OlapBuilderGoogleBigqueryDataSource) table.getDataSource();

        final String parentProjectId = ds.getCredentialsProjectId();
        final TableId apiTableId = table.getApiTableId(connection);

        final String dsProjectId = apiTableId.getProject();
        final String dataSetId = apiTableId.getDataset();
        final String tableId = apiTableId.getTable();

        tableNameQ = dsProjectId + "." + dataSetId + "." + tableId;

        OlapLoggers.BUILDER_GOOGLE.debug("[big-query-storage] init session for table [" + tableNameQ + "]");

        final String parent = String.format("projects/%s", parentProjectId);
        final String srcTable = apiTableId.getIAMResourceName();

        final Credentials credentials = connection.getCredentials();

        try
        {
            final BigQueryReadSettings settings = BigQueryReadSettings.newBuilder()
                    .setCredentialsProvider(() -> credentials)
                    .build();

            client = BigQueryReadClient.create(settings);

            final List<IOlapBuilderDataColumnDef> columns = table.getSelectedColumns();

            final ReadSession.TableReadOptions.Builder optionsBuilder = ReadSession.TableReadOptions.newBuilder();

            for (IOlapBuilderDataColumnDef column : columns)
            {
                optionsBuilder.addSelectedFields(column.getName());
            }

            final List<String> restrictions = table.getRestrictionList();

            if (restrictions != null)
            {

                for (String restriction : restrictions)
                {
                    optionsBuilder.setRowRestriction(restriction);
                }
            }

            final Comparable incrLoadMarker = context.getIncrementalLoadMarker(table);

            if (incrLoadMarker != null)
            {
                final String restriction = table.setupIncrLoadRestriction(tableNameQ, incrLoadMarker);

                optionsBuilder.setRowRestriction(restriction);

                OlapLoggers.BUILDER_GOOGLE_BQS.debug(String.format(
                        "[big-query-storage] table [%s] incr. load restriction [%s]", tableNameQ, restriction
                ));
            }

            final ReadSession.Builder sessionBuilder = ReadSession.newBuilder()
                    .setTable(srcTable)
                    .setDataFormat(DataFormat.ARROW)
                    .setReadOptions(optionsBuilder);

            final CreateReadSessionRequest.Builder builder =
                    CreateReadSessionRequest.newBuilder()
                            .setParent(parent)
                            .setReadSession(sessionBuilder);

            if (maxRowCount != -1)
            {
                builder.setMaxStreamCount(1);
            }

            final ReadSession session = client.createReadSession(builder.build());

            final int streamCount = session.getStreamsCount();

            OlapLoggers.BUILDER_GOOGLE.debug("[big-query-storage] table [" + tableNameQ + "] session stream count:" + streamCount);

            final ServerStream<ReadRowsResponse>[] streams = new ServerStream[streamCount];

            try
            {
                for (int streamIndex = 0; streamIndex < streams.length; streamIndex++)
                {
                    final String streamName = session.getStreams(streamIndex).getName();

                    final ReadRowsRequest readRowsRequest = ReadRowsRequest.newBuilder()
                            .setReadStream(streamName)
                            .build();

                    final ServerStreamingCallable<ReadRowsRequest, ReadRowsResponse> callable = client.readRowsCallable();
                    final ServerStream<ReadRowsResponse> stream = callable.call(readRowsRequest);

                    streams[streamIndex] = stream;
                }

                this.streams = new OlapBuilderGoogleBigQueryStorageStreams(tableNameQ, session.getArrowSchema(), streams);

            }
            catch (RuntimeException ex)
            {
                OlapLoggers.BUILDER_GOOGLE_BQS.error("[big-query-storage] table [" + tableNameQ + "] error (cancelling created streams)", ex);

                cancelCreatedStreamsSilently(streams);

                throw ex;
            }

        }
        catch (IOException ex)
        {
            throw new OlapBuilderErrorException(ex, OlapGoogleApiError.IO_ERROR, ex.getLocalizedMessage());
        }
    }

    private void cancelCreatedStreamsSilently(ServerStream<ReadRowsResponse>[] streams)
    {
        for (int ii = 0; ii < streams.length; ii++)
        {
            final ServerStream<ReadRowsResponse> stream = streams[ii];

            if (stream != null)
            {
                try
                {
                    stream.cancel();
                }
                catch (RuntimeException ex)
                {
                    OlapLoggers.BUILDER_GOOGLE_BQS.error("[big-query-storage] table [" + tableNameQ + "] error while cancelling the created stream[" + ii + "]", ex);
                }
            }
        }
    }

    @Override
    public boolean isRowSafe()
    {
        return false;
    }

    @Override
    public IOlapBuilderTableRow doNextRow()
    {
        if (maxRowCount != -1 && rowsCount >= maxRowCount)
        {
            return null;
        }

        if (this.streams == null)
        {
            return null;
        }

        try
        {
            final IOlapBuilderTableRow row = this.streams.nextRow();

            if (row == null)
            {
                return null;
            }

            rowsCount++;

            return row;

        }
        catch (IOException ex)
        {
            throw new OlapBuilderErrorException(ex, OlapGoogleApiError.IO_ERROR, ex.getLocalizedMessage());
        }
    }

    /**
     * Will be called as soon as the init method has been called (either successfully or on error).
     * Keep it safe as it is typically called from a finally block.
     */
    @Override
    public void done()
    {
        OlapLoggers.BUILDER_GOOGLE.debug("[big-query-storage] closing table row reader [" + (tableNameQ != null ? tableNameQ : table.getName()) + "]");

        if (streams != null)
        {
            try
            {
                streams.close();
                streams = null;
            }
            catch (RuntimeException ex)
            {
                OlapLoggers.BUILDER_GOOGLE_BQS.info("[big-query-storage] could not close the streams", ex);
            }
        }

        if (client != null)
        {
            try
            {
                client.close();
                client = null;
            }
            catch (RuntimeException ex)
            {
                OlapLoggers.BUILDER_GOOGLE_BQS.info("[big-query-storage] could not close the client", ex);
            }
        }

        super.done();
    }


}
