## Java (ETL)

The Java transformation allows for defining your own transformation logic using the Java language. The advantage
of using Java code is not only all the power of Java and the libraries but that the code is going to be as fast
as icCube code as it is compiled and injected into icCube.

![java](img/etl_java_large.png)

The transformation defines a class that has to implement the interface `IOlapBuilderViewLogic`
(don't change the name of your class).

```java
/**
 * The interface for a Java View
 */
public interface IOlapBuilderViewLogic
{
    /**
     * @return false if you want to manage the cache on your side (row won't be cached)
     */
    default boolean onNewCachedRow(IOlapDataTableDef tableDef, IOlapReadOnlyDataRow row)
    {
        return true;
    }

    /**
     * Called once the cached tables have been loaded and before the main table load starts
     *
     * @param cachedTables a map of cached tables, if any, with the data already cached
     * @param mainTable    the main table definition
     */
    void onInitMainTable(Map<String, IOlapCachedTable> cachedTables, IOlapDataTableDef mainTable);

    /**
     * Called for each row read on the main table (not thread safe). Called after onInitMainTable
     *
     * @param context      a context object allowing to fire rows.
     * @param mainTableRow row of the mainTable that can not be cached internally as there is no guarantee if will be available after this method is called
     *
     * @return true to continue reading the main table
     */
    boolean onNewRow(IOlapViewContext context, Map<String, IOlapCachedTable> cachedTables, IOlapDataTableDef mainTable, IOlapReadOnlyDataRow mainTableRow);

    /**
     * Called once when all rows of all tables have been read and after onNewRow. You can fire rows here if needed
     */
    void onProcessingCompleted(IOlapViewContext context, Map<String, IOlapCachedTable> cachedTables);

    /**
     * @return true if onNewRow() should be, if necessary, synchronized externally
     */
    default boolean synchronizeOnRow()
    {
        return true;
    }
}
```

Hint : You can complete the variables (e.g. row, context..., with `CTRL+SPACE`).

If you implement the method `synchronizeOnRow()` and make it return false you are responsible for making
the `onNewRow()` thread safe if `Table Row Ordering` is set to `None required`. By default, it will be
synchronized if required by the `Table Row Ordering`.

### Working with Intellij IDEA (Eclipse, Visual Studio, ...)

You can develop the class outside icCube, using for example IntelliJ IDEA. For this you can create an empty project
and include all Java libraries found under `${icCube-install}/lib`. Then you can create your class implementing the
`IOlapBuilderViewLogic` interface from `crazydev.iccube.pub.view` package. Once the class is running as expected you
can then either copy/paste or add this class in the jar file that you copy into the `/lib` directory.

### Existing Class Mode

If you've a class that implements the `IOlapBuilderViewLogic` interface and has been included in the project, just
move the jar file into the `${icCube-install}\lib` directory. You can directly use it without the need for the JDK
compiler. For this you need to change the java body with a single line with your java class name (full qualified name).
For example, as in our example attached `crazydev.iccube.pub.view.InterCompanyElimination`. The drawback is indeed
the hard coded configuration.

### Support

Do not hesitate to [contact us](https://www.icCube.com/contact) for more help on this.

### Download

As an example we implemented a view that calculated an InterCompany Elimination ([www](data/Walkthrough_ETL_Java.zip)).

Next chapter: [GeoCoding](etl_geocoding.md) describes how to do implement a GeoCoding transformation that retrieves latitude and longitude from addresses.

_
