## Configuration File : icCube-notification.xml

icCube allows for being notified about various internal activities. For example, administrators may be
notified via eMail that a schema has been reloaded or updated. This service is configured via the
`icCube-notification.xml` file.

### Java System Properties

A Java system property may be used to define a configuration.

For example, the following string :

    ${http.port:-8282}      or  ${sys:http.port:-8282} 

will be replaced by the value of the `http.port` variable or `8282` if the variable is not defined.

### Environment Variables

Environment variables can be used wherever a Java system property can be used.

For example, the following string :

    ${env:SERVER:-localhost}

will be replaced by the value of the `SERVER` variable or `localhost` if the variable is not defined.

### Configuration Overview

Notifications are configured via a list of notification listeners. Each listener is filtering
notifications (e.g., errors vs warnings) to be dispatched to a given transport service (e.g., eMail).

Here is an example of configuration that is reporting all internal errors to the icCube support
(hopefully none.) Note that up-to-date information is available in the configuration file delivered
with icCube:

```xml

<icCubeNotificationConfiguration>

    <listener>

        <filter>
            <type>INTERNAL_ERROR</type>
        </filter>

        <transporter>

            <id>icCube Support</id>

            <service>eMail</service>

            <param>
                <name>from</name>
                <value>icCube@yourDomain.com</value>
            </param>

            <param>
                <name>to</name>
                <value>support@icCube.com</value>
            </param>

        </transporter>

    </listener>

</icCubeNotificationConfiguration>
```

### Filter

Each listener can define a filter to decide which notification is going to be dispatched; no filter meaning
all notifications are accepted by this listener. Three fields are available:

| Field    | Description                                                         |
|----------|---------------------------------------------------------------------|
| `level`  | the severity level of the notification (`INFO`, `WARNING`, `ERROR`) |
| `type`   | e.g., `INTERNAL_ERROR`, `SCHEMA_FULL_LOAD`, `SCHEMA_UNLOAD`, etc... |
| `schema` | the name of the schema the notification is attached to              |

Each field accepts [JAVA regular expressions](http://docs.oracle.com/javase/6/docs/api/java/util/regex/Pattern.html).
For example, the following filter is accepting warning and error notifications:

```xml

<filter>
    <level>WARNING|ERROR</level>
</filter>
```

This filter is accepting all warnings about schema life cycle activity:

```xml

<filter>
    <level>WARNING.*</level>
    <type>SCHEMA.*</type>
</filter>
```

### Message Types

The following types of notifications are available and can be used in the `type` field of the filters:

```
INTERNAL_ERROR
    An unexpected error within the system (typically reported to icCube support).

LICENSE_EXPIRY_WARNING
    The license is about to expire ( 30 - 15 - 1 day warnings ).

SERVER_READY
    The icCube server has been successfully started and is ready.

SCHEMA_FULL_LOAD
    A full schema load (vs. incremental load).

SCHEMA_INCREMENTAL_LOAD
    A schema has been successfully updated (vs. full load).

SCHEMA_INCREMENTAL_LOAD_DEACTIVATED
    A schema update (vs. full load) has done nothing due to incremental load being de-activated
    since a previous incremental load on error.

SCHEMA_INCREMENTAL_LOAD_ERROR
    A schema has been unsuccessfully incremental loaded (the schema is in an uncertain state).

SCHEMA_FULL_LOAD_ERROR
    A schema has been unsuccessfully fully loaded.

SCHEMA_LOAD_ERROR
    A schema has been unsuccessfully loaded (could not determine if incremental load or full load).

SCHEMA_LOAD_CANCELLED
    A schema load request has been cancelled.

SCHEMA_UNLOAD
    A schema has been successfully unloaded.

SCHEMA_PARTITIONS_UNLOADED
    Schema: unload partitions successful.

SCHEMA_PARTITIONS_LOADED
    Schema: load partitions successful.

SCHEMA_PARTITIONS_LOAD_FAILED
    Schema: load partitions on error: the schema state is still consistent
    (e.g., SELECT error, authorization error, etc...).

SCHEMA_PARTITIONS_LOAD_ERROR
    Schema: load partitions on error: the state of the schema is uncertain.

ALERT
    An alert has been triggered and is positive.
```

### Transport Services

Along with its filter, a listener is deciding which transport service is going to deliver the notifications.
icCube is currently providing an eMail service (via the configuration of an outgoing SMTP server); have a look
to the next paragraph to see how to implement your own service. The following is demonstrating how to configure
a GMail SMTP server via SSL. The icCube implementation is based on JavaMail and accept all the parameters
as defined by this standard; the `icCube.username` and `password` are the (optional) credentials required to
connect to your SMTP server:

```xml

<transportService>

    <id>eMail</id>

    <service-class>crazydev.iccube.notification.transporter.email.OlapNotificationEmailJavaMailService</service-class>

    <param>
        <name>icCube.username</name>
        <value>...</value>
    </param>

    <param>
        <name>icCube.password</name>
        <value>...</value>
    </param>


    <param>
        <name>mail.smtp.host</name>
        <value>smtp.gmail.com</value>
    </param>

    <param>
        <name>mail.smtp.socketFactory.port</name>
        <value>465</value>
    </param>

    <param>
        <name>mail.smtp.socketFactory.class</name>
        <value>javax.net.ssl.SSLSocketFactory</value>
    </param>

    <param>
        <name>mail.smtp.auth</name>
        <value>true</value>
    </param>

    <param>
        <name>mail.smtp.port</name>
        <value>465</value>
    </param>

</transportService>
```

### User Defined Transport Service

The configuration of the notification service is fully generic and you can provide your own transport
services. For that purpose you need to provide a JAVA class that implements the interface:

`crazydev.iccube.notification.transporter.OlapNotificationTransportService`

and specify this class name in the `service-class` configuration field of the `transportService`.

As an example, the following code is implementing a transport service that keeps notifications in memory:

```java
public class InMemoryTransportService implements OlapNotificationTransportService {
    private final List<OlapNotification> notifications = new ArrayList<OlapNotification>();

    private String id;

    public TransportService() {
    }

    @Override
    public String getId() {
        return this.id;
    }

    @Override
    public void configure(OlapNotificationTransportServiceDefinition configuration)
            throws OlapNotificationServiceException {
        this.id = configuration.getId();
    }

    @Override
    public OlapNotificationServiceTransporter transporter(OlapNotificationTransporterDefinition configuration)
            throws OlapNotificationServiceException {
        return new Transporter(this, configuration.getId());
    }

    public void handleNotification(OlapNotification notification) {
        notifications.add(notification);
    }

}

public class Transporter extends OlapNotificationServiceTransporter {
    private final InMemoryTransportService service;

    private final String id;

    public Transporter(InMemoryTransportService service, String id) {
        this.service = service;
        this.id = id;
    }

    @Override
    public String getId() {
        return id;
    }

    @Override
    public String asLogInfo() {
        return id;
    }

    @Override
    public void handleNotification(OlapNotification notification) {
        service.handleNotification(notification);
    }

}
```

and the configuration of a listener dispatching all the notifications to this transport service is as follows:

```xml

<icCubeNotificationConfiguration>

    <listener>

        <transporter>

            <id>Test</id>
            <service>In Memory</service>

        </transporter>

    </listener>

    <transportService>

        <id>In Memory</id>
        <service-class>InMemoryTransportService</service-class>

    </transportService>

</icCubeNotificationConfiguration>
```

Next chapter : [Environment Variables](env_variables.md).

_