## MDX FAQ : How to generate SQL-like table result?

MDX is a powerful language, but it has some pitfalls that can affect performance. A common misconception is that
a crossjoin can be used to produce an SQL-like table. Although the crossjoin can produce the exact table you want,
it might come at performance cost - especially as your model scales. Fortunately, MDX provides a more efficient
alternative.

Let's look at a retail example to illustrate this :

    Dimension     : [Item]          listing all the available items according to different hierarchies
        Hierarchy : [Item]
        Hierarchy : [Department]
        Hierarchy : [Brand]
        Hierarchy : [SKU]

In real-world scenarios, models often include 20-40 hierarchies in the `[Item]` dimension, some with multiple levels.

In this example, you want to display a single measure - the number of sold units -  for a selected list of items, 
along with their department, brand and SKU. This can be achieved as follows :

    select

      [Measures].[#Unit] on 0

      non empty 

        [Item].[Department].[Department]
        * [Item].[Brand].[Brand]
        * [Item].[Item].[Item]
        * [Item].[SKU].[SKU]

        on 1

    filterby [Time].[Calendar].[2025]

To keep the query simple, we'll focus on the `[Items]` crossjoin used to retrieve several _properties_ of the
selected items. As expected, this query returns the correct results :

| Department | Brand  | Item | SKU          | #Unit |
|------------|--------|------|--------------|------:|
| Cars       | Toyota | CHR  | ABC-12345-BL |    42 |
| ...        | ...    | ...  | ...          |    .. |

Let's see what is actually happens under the hood when icCube compute the crossjoin.

icCube must compute the sub-cube containing all possible combinations (aka. Cartesian product) of members
from the four sets. Finally, it filters out empty combinations using `[#Unit]` measure. In real-word scenarios,
the Cartesian explosion makes the number of combinations grow exponentially. Fortunately, icCube recognizes
that not all the combinations are needed - especially when the sets come from the same dimension. Still,
this optimization is no trivial task. Therefore, **use caution when writing crossjoin in large models**.

Fortunately, MDX and icCube provide a more efficient way to retrieve the properties of the selected items.

Note that department, brand and SKU are **not needed to define the item set** - only `[Item].[Item].[Item]`,
along with `non empty` and `filterby` conditions, is required. In most cases, item members include several
related properties - or better yet, these are modeled as attributes of the items.

What if we let the crossjoin handle the item selection, and then use a dedicated syntax to efficiently
retrieve properties and attributes of the items ? This can be achieved as follows :

    select

      [Measures].[#Unit] on 0

      non empty [Item].[Item].[Item] on 1

    filterby [Time].[Calendar].[2025]

    TidyPostProcessor DimensionProperties(
      _( [Item].[Item].[Departement], 1, 0 ),
      _( [Item].[Item].[Brand],       1, 1 ),
      _( [Item].[Item].[SKU],         1, 3 )
    )

Now, the axis 1 contains a single set and the added [DimensionProperties](../post_processing/DimensionProperties.md)
post-processor retrieves the properties and automatically add them as columns to the result. In this example, the
crossjoin is eliminated entirely - making the query runs much faster.

## See Also

- [Dimension Properties Post-Processor](../post_processing/DimensionProperties.md)
- [Dimension Properties](../mdx/Dimension%20Properties.md)
- [MDX FAQ](mdx_faq.md)

_