## FORMAT_STRING

Controls how cell values are formatted.




### Description

icCube supports four different ways to format values:

<b>a) Named Format</b>
<ul>
<li>Fixed         : Displays at least one digit to the left and two digits to the right of the decimal separator.
<li>Percent       : Displays the number multiplied by 100 with a percent sign (%) appended to the right. Always displays two digits to the right of the decimal separator.
<li>Standard      : Displays the number with a thousand separator, at least one digit to the left and two digits to the right of the decimal separator.
<li>Currency      : Displays the number with a thousand separator, if appropriate. Displays two digits to the right of the decimal separator. Output is based on system locale settings.
<li>Scientific    : Uses standard scientific notation.
<li>Yes/No        : Displays No if the number is 0; otherwise, displays Yes.
<li>On/Off        : Displays Off if the number is 0; otherwise, displays On.
<li>True/False    : Displays False if the number is 0; otherwise, displays True.
<li>ElapsedMillis : Formats millisecond values and display hours, minutes, seconds and milliseconds.
</ul>
The following example is formatting the values according to the 'Percent' named format:
<pre>
WITH MEMBER val AS mdx-expression, FORMAT_STRING = 'Percent'
</pre>

<b>b) Excel/VB - MDX Standard Syntax</b>

This is the default format in icCube (for versions greater than 2.5.4).

More details can be found [here](http://msdn.microsoft.com/en-us/library/ms146084.aspx). Use this format if you plan to use Excel (or any other client tool that can use `FORMAT_STRING` values).

<pre>
WITH MEMBER val AS mdx-expression, FORMAT_STRING = "#,##0.0;(#,##0.0),0"
</pre>

The usage of Excel/VB syntax can be forced with the 'e:' prefix :

<pre>
WITH MEMBER val AS mdx-expression, FORMAT_STRING = "e:#,##0.0;(#,##0.0),0"
</pre>

<b>c) JAVA Syntax</b>

Numerical values are formatted using JAVA format ([www](https://docs.oracle.com/en/java/javase/25/docs/api/java.base/java/util/Formatter.html)):

<pre>
WITH MEMBER val AS mdx-expression, FORMAT_STRING = '%3.2f'
</pre>

Date values are formatted using JAVA simple date format ([www](https://docs.oracle.com/en/java/javase/25/docs/api/java.base/java/text/SimpleDateFormat.html)):

<pre>
WITH MEMBER val AS mdx-expression, FORMAT_STRING = 'MMM dd, yyyy'
</pre>

The usage of Java syntax can be forced with the 'j:' prefix :

<pre>
WITH MEMBER val AS mdx-expression, FORMAT_STRING = 'j:%3.2f'
</pre>

<b>d) Function : formatter()</b>

The function formatter('name', 'pattern') allows for formatting MDX entities. For example, "duration" allows to format a duration value expressed in milliseconds (e.g. 23'000 milliseconds):

<pre>
WITH MEMBER val AS mdx-expression, FORMAT_STRING = formatter("duration", "%h h. %m min.")
</pre>

<b>currentCellValue()</b>

The function currentCellValue() allows for creating FORMAT_STRING depending on the value of the cell being formatted (conditional formatting):

<pre>
FORMAT_STRING = iif( isNaN( currentCellValue() ),
                      'invalid number',
                      iif( isInfinity( currentCellValue() ), 'infinite number', '%f' )
                )
</pre>

<b>e) Function : WithFormattedValue()</b>

The MDX function WithFormattedValue() can be used to attach a formatted value to an existing scalar entity.
For example, the following declared function withFV() is formatting an amount using a currency name :

<pre>
CREATE NATIVE FUNCTION RpJavaFormat( String locale, double amount, String curr ) AS
/* JAVA : String

    import crazydev.common.utils.CdLocaleUtils;

    return String.format(CdLocaleUtils.fromName(locale, Locale.getDefault()), "%,.2f", amount) + " " + curr;

*/

CREATE FUNCTION withFV( _v, _curr ) as _v.withFormattedValue(
    RpJavaFormat(UserLocale(), _v , _curr)
)
</pre>

Ensure to call this function right before returning the value of a calculated member and to define
a fake FORMAT_STRING value to trigger the processing of the formatted value :

<pre>
CREATE MEMBER XX as withFV( ... ), FORMAT_STRING = "Embedded$"
</pre>

<b>Configuration</b>

icCube supports both JAVA and Excel/VB like (i.e., MDX compliant) syntax for defining a `FORMAT_STRING`. The default syntax is configured in the file [icCube.xml](../../user_guide/configuring_iccube/iccube_xml.md#format_string) using the property :

<pre>
icCube.defaultFormatStringSyntax   -- value=EXCEL|JAVA  default:EXCEL
</pre>



             
### Examples

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Fixed (1)

</div>

<div class="ic3-mdx-example-body" style="display:none">

The named format 'Fixed' is formatting values as a percent. Displays at least one digit to the left and two digits to the right of the decimal separator.

**Query**

```
with member val_ as 50 / 100, FORMAT_STRING = 'Fixed'
select from sales where val_ cell properties formatted_value
```

 
**Result**

<pre>0.50</pre>



**Assertion : MDX Equals**

```
with member val_ as "0.50"
select from sales where val_ cell properties formatted_value
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Fixed (2)

</div>

<div class="ic3-mdx-example-body" style="display:none">

The named format 'Fixed' is formatting values as a percent. Displays at least one digit to the left and two digits to the right of the decimal separator.

**Query**

```
with member val_ as 50, FORMAT_STRING = 'Fixed'
select from sales where val_ cell properties formatted_value
```

 
**Result**

<pre>50.00</pre>



**Assertion : MDX Equals**

```
with member val_ as "50.00"
select from sales where val_ cell properties formatted_value
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Percent (1)

</div>

<div class="ic3-mdx-example-body" style="display:none">

The named format 'Percent' is formatting values as a percent. Displays the number multiplied by 100 with a percent sign (%) appended to the right. Always displays two digits to the right of the decimal separator.

**Query**

```
with member val_ as 50 / 100, FORMAT_STRING = 'Percent'
select from sales where val_ cell properties formatted_value
```

 
**Result**

<pre>50.00%</pre>



**Assertion : MDX Equals**

```
with member val_ as "50.00%"
select from sales where val_ cell properties formatted_value
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Percent (2)

</div>

<div class="ic3-mdx-example-body" style="display:none">

The named format 'Percent' is formatting values as a percent. Displays the number multiplied by 100 with a percent sign (%) appended to the right. Always displays two digits to the right of the decimal separator.

**Query**

```
with member val_ as 50, FORMAT_STRING = 'Percent'
select from sales where val_ cell properties formatted_value
```

 
**Result**

<pre>5000.00%</pre>



**Assertion : MDX Equals**

```
with member val_ as "5000.00%"
select from sales where val_ cell properties formatted_value
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Standard

</div>

<div class="ic3-mdx-example-body" style="display:none">

The named format 'Standard' Displays the number with thousand separator, at least one digit to the left and two digits to the right of the decimal separator.

**Query**

```
WITH
    MEMBER [1500] AS 1500, FORMAT_STRING = 'Standard'
    MEMBER [15001500] AS 15001500, FORMAT_STRING = 'Standard'
    MEMBER [0.315] AS 0.315, FORMAT_STRING = 'Standard'
    MEMBER [0.314] AS 0.314, FORMAT_STRING = 'Standard'
    MEMBER [-0.314] AS -0.314, FORMAT_STRING = 'Standard'
    MEMBER [-0.315] AS -0.315, FORMAT_STRING = 'Standard'
SELECT
    {[1500],[15001500],[0.315],[0.314],[-0.314],[-0.315]} on 0
FROM sales
CELL PROPERTIES formatted_value
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">1500</td>
  <td class="ic3-mdx-result-cell-header">15001500</td>
  <td class="ic3-mdx-result-cell-header">0.315</td>
  <td class="ic3-mdx-result-cell-header">0.314</td>
  <td class="ic3-mdx-result-cell-header">-0.314</td>
  <td class="ic3-mdx-result-cell-header">-0.315</td>
</tr>
<tr>
 <td>1,500.00</td>
 <td>15,001,500.00</td>
 <td>0.32</td>
 <td>0.31</td>
 <td>-0.31</td>
 <td>-0.32</td>
</tr>
</table>



**Assertion : MDX Equals**

```
WITH
    MEMBER [1500] AS "1,500.00"
    MEMBER [15001500] AS "15,001,500.00"
    MEMBER [0.315] AS "0.32"
    MEMBER [0.314] AS "0.31"
    MEMBER [-0.314] AS "-0.31"
    MEMBER [-0.315] AS "-0.32"
SELECT
    {[1500],[15001500],[0.315],[0.314],[-0.314],[-0.315]} on 0
FROM sales
CELL PROPERTIES formatted_value
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Currency

</div>

<div class="ic3-mdx-example-body" style="display:none">

The named format 'Currency' displays the number with a thousand separator and the currency. Displays two digits to the right of the decimal separator. Note that the default negative currency format has changed since Java 11: the negative sign is used instead of parenthesis (see [this page](https://docs.oracle.com/en/java/javase/17/docs/api/java.base/java/text/NumberFormat.html#getCurrencyInstance(java.util.Locale)) for more details).

**Query**

```
WITH
    MEMBER [1500] AS 1500, FORMAT_STRING = 'Currency'
    MEMBER [15001500] AS 15001500, FORMAT_STRING = 'Currency'
    MEMBER [0.315] AS 0.315, FORMAT_STRING = 'Currency'
    MEMBER [0.314] AS 0.314, FORMAT_STRING = 'Currency'
    MEMBER [-0.314] AS -0.314, FORMAT_STRING = 'Currency'
    MEMBER [-0.315] AS -0.315, FORMAT_STRING = 'Currency'
SELECT
    {[1500],[15001500],[0.315],[0.314],[-0.314],[-0.315]} on 0
FROM sales
CELL PROPERTIES formatted_value
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">1500</td>
  <td class="ic3-mdx-result-cell-header">15001500</td>
  <td class="ic3-mdx-result-cell-header">0.315</td>
  <td class="ic3-mdx-result-cell-header">0.314</td>
  <td class="ic3-mdx-result-cell-header">-0.314</td>
  <td class="ic3-mdx-result-cell-header">-0.315</td>
</tr>
<tr>
 <td>$1,500.00</td>
 <td>$15,001,500.00</td>
 <td>$0.32</td>
 <td>$0.31</td>
 <td>-$0.31</td>
 <td>-$0.32</td>
</tr>
</table>



**Assertion : MDX Equals**

```
WITH
    MEMBER [1500] AS "$1,500.00"
    MEMBER [15001500] AS "$15,001,500.00"
    MEMBER [0.315] AS "$0.32"
    MEMBER [0.314] AS "$0.31"
    MEMBER [-0.314] AS "-$0.31"
    MEMBER [-0.315] AS "-$0.32"
SELECT
    {[1500],[15001500],[0.315],[0.314],[-0.314],[-0.315]} on 0
FROM sales
CELL PROPERTIES formatted_value
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Scientific

</div>

<div class="ic3-mdx-example-body" style="display:none">

The named format 'Scientific' displays the number using standard scientific notation.

**Query**

```
WITH
    MEMBER [1500] AS 1500, FORMAT_STRING = 'Scientific'
    MEMBER [15001500] AS 15001500, FORMAT_STRING = 'Scientific'
    MEMBER [0.315] AS 0.315, FORMAT_STRING = 'Scientific'
    MEMBER [0.314] AS 0.314, FORMAT_STRING = 'Scientific'
    MEMBER [-0.314] AS -0.314, FORMAT_STRING = 'Scientific'
    MEMBER [-0.315] AS -0.315, FORMAT_STRING = 'Scientific'
SELECT
    {[1500],[15001500],[0.315],[0.314],[-0.314],[-0.315]} on 0
FROM sales
CELL PROPERTIES formatted_value
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">1500</td>
  <td class="ic3-mdx-result-cell-header">15001500</td>
  <td class="ic3-mdx-result-cell-header">0.315</td>
  <td class="ic3-mdx-result-cell-header">0.314</td>
  <td class="ic3-mdx-result-cell-header">-0.314</td>
  <td class="ic3-mdx-result-cell-header">-0.315</td>
</tr>
<tr>
 <td>1.50E3</td>
 <td>1.50E7</td>
 <td>3.15E-1</td>
 <td>3.14E-1</td>
 <td>-3.14E-1</td>
 <td>-3.15E-1</td>
</tr>
</table>



**Assertion : MDX Equals**

```
WITH
    MEMBER [1500] AS "1.50E3"
    MEMBER [15001500] AS "1.50E7"
    MEMBER [0.315] AS "3.15E-1"
    MEMBER [0.314] AS "3.14E-1"
    MEMBER [-0.314] AS "-3.14E-1"
    MEMBER [-0.315] AS "-3.15E-1"
SELECT
    {[1500],[15001500],[0.315],[0.314],[-0.314],[-0.315]} on 0
FROM sales
CELL PROPERTIES formatted_value
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Yes/No

</div>

<div class="ic3-mdx-example-body" style="display:none">

The named format 'Yes/No' displays No if the number is 0; otherwise, displays Yes.

**Query**

```
WITH
    MEMBER [Yes] AS 1, FORMAT_STRING = 'Yes/No'
    MEMBER [YES (upper)] AS 23, FORMAT_STRING = 'YES/No' -- you can play with upper/lowercase
    MEMBER [Yes (true)] AS true, FORMAT_STRING = 'Yes/No'
    MEMBER [No] AS 0, FORMAT_STRING = 'Yes/No'
SELECT
    {[Yes],[YES (upper)],[Yes (true)],[No]} on 0
FROM sales
CELL PROPERTIES formatted_value
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">Yes</td>
  <td class="ic3-mdx-result-cell-header">YES (upper)</td>
  <td class="ic3-mdx-result-cell-header">Yes (true)</td>
  <td class="ic3-mdx-result-cell-header">No</td>
</tr>
<tr>
 <td>Yes</td>
 <td>YES</td>
 <td>Yes</td>
 <td>No</td>
</tr>
</table>



**Assertion : MDX Equals**

```
WITH
    MEMBER [Yes] AS "Yes"
    MEMBER [YES (upper)] AS "YES"
    MEMBER [Yes (true)] AS "Yes"
    MEMBER [No] AS "No"
SELECT
    {[Yes],[YES (upper)],[Yes (true)],[No]} on 0
FROM sales
CELL PROPERTIES formatted_value
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### True/False

</div>

<div class="ic3-mdx-example-body" style="display:none">

The named format 'Yes/No' displays False if the number is 0; otherwise, displays True.

**Query**

```
WITH
    MEMBER [True] AS 1, FORMAT_STRING = 'True/False'
    MEMBER [TRUE (upper)] AS 23, FORMAT_STRING = 'TRUE/False' -- you can play with upper/lowercase
    MEMBER [True (true)] AS true, FORMAT_STRING = 'True/False'
    MEMBER [False] AS 0, FORMAT_STRING = 'True/False'
SELECT
    {[True],[TRUE (upper)],[True (true)],[False]} on 0
FROM sales
CELL PROPERTIES formatted_value
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">True</td>
  <td class="ic3-mdx-result-cell-header">TRUE (upper)</td>
  <td class="ic3-mdx-result-cell-header">True (true)</td>
  <td class="ic3-mdx-result-cell-header">False</td>
</tr>
<tr>
 <td>True</td>
 <td>TRUE</td>
 <td>True</td>
 <td>False</td>
</tr>
</table>



**Assertion : MDX Equals**

```
WITH
    MEMBER [True] AS "True"
    MEMBER [TRUE (upper)] AS "TRUE"
    MEMBER [True (true)] AS "True"
    MEMBER [False] AS "False"
SELECT
    {[True],[TRUE (upper)],[True (true)],[False]} on 0
FROM sales
CELL PROPERTIES formatted_value
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### On/Off

</div>

<div class="ic3-mdx-example-body" style="display:none">

The named format 'On/Off' displays Off if the number is 0; otherwise, displays On.

**Query**

```
WITH
    MEMBER [On] AS 1, FORMAT_STRING = 'On/Off'
    MEMBER [on (lower)] AS 23, FORMAT_STRING = 'on/Off' -- you can play with upper/lowercase
    MEMBER [On (true)] AS true, FORMAT_STRING = 'On/Off'
    MEMBER [Off] AS 0, FORMAT_STRING = 'On/Off'
SELECT
    {[On],[on (lower)],[On (true)],[Off]} on 0
FROM sales
CELL PROPERTIES formatted_value
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">On</td>
  <td class="ic3-mdx-result-cell-header">on (lower)</td>
  <td class="ic3-mdx-result-cell-header">On (true)</td>
  <td class="ic3-mdx-result-cell-header">Off</td>
</tr>
<tr>
 <td>On</td>
 <td>on</td>
 <td>On</td>
 <td>Off</td>
</tr>
</table>



**Assertion : MDX Equals**

```
WITH
    MEMBER [On] AS "On"
    MEMBER [on (lower)] AS "on"
    MEMBER [On (true)] AS "On"
    MEMBER [Off] AS "Off"
SELECT
    {[On],[on (lower)],[On (true)],[Off]} on 0
FROM sales
CELL PROPERTIES formatted_value
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### ElapsedMillis

</div>

<div class="ic3-mdx-example-body" style="display:none">

The named format 'ElapsedMillis' is formatting millisecond values and display hours, minutes, seconds and milliseconds.

**Query**

```
with member val_ as ((3600 + 60 + 2) * 1000) + 432, FORMAT_STRING = 'ElapsedMillis'
select from sales where val_ cell properties formatted_value
```

 
**Result**

<pre>01h 01m 02s 432ms</pre>



**Assertion : MDX Equals**

```
with member val_ as "01h 01m 02s 432ms"
select from sales where val_ cell properties formatted_value
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Excel/VB : Number Formatting

</div>

<div class="ic3-mdx-example-body" style="display:none">

Some examples of number formatting following MDX standard notation.

**Query**

```
WITH
     MEMBER [--- NUMERIC ---] AS ""
     MEMBER ['#####.##'   --> 8315.4] AS 8315.4, FORMAT_STRING = '#####.##'
     MEMBER ['##,###.##'  --> 8315.4] AS 8315.4, FORMAT_STRING = '##,###.##'
     MEMBER ['00000.00'   --> 8315.4] AS 8315.4, FORMAT_STRING = '00000.00'
     MEMBER ['#####0.0##' --> 8315.4] AS 8315.4, FORMAT_STRING = '#####0.0##'
     MEMBER ['$0'         --> 8315.4] AS 8315.4, FORMAT_STRING = '$0'
     MEMBER [--- PERCENTAGE ---] AS ""
     MEMBER ['0%'   --> 0.4231] AS 0.4231, FORMAT_STRING = '0%'
     MEMBER ['0.0%' --> 0.4231] AS 0.4231, FORMAT_STRING = '0.0%'
     MEMBER [--- SCIENTIFIC ---] AS ""
     MEMBER ['0.00E+00' --> 42.31] AS 42.31, FORMAT_STRING = '0.00E+00'
     MEMBER [--- ROUNDING ---] AS ""
     MEMBER ['0.0' -->  5.149] AS 5.149, FORMAT_STRING =  '0.0'
     MEMBER ['0.0' -->  5.150] AS 5.150, FORMAT_STRING =  '0.0'
     MEMBER ['0.0' --> -5.149] AS -5.149, FORMAT_STRING =  '0.0'
     MEMBER ['0.0' --> -5.150] AS -5.150, FORMAT_STRING =  '0.0'
     MEMBER [--- SECTIONS ---] AS ""
     MEMBER ['0.#;Neg (0.#);0.00' --> +8315.115] AS 8315.115, FORMAT_STRING =  '0.#;Neg (0.#);0.00;Nothing'
     MEMBER ['0.#;Neg (0.#);0.00' --> -8315.115] AS -8315.115, FORMAT_STRING =  '0.#;Neg (0.#);0.00;Nothing'
     MEMBER ['0.#;Neg (0.#);0.00' --> 0.0] AS 0, FORMAT_STRING =  '0.#;Neg (0.#);0.00;Nothing'
     MEMBER [--- Divide by 1000 with , and multiply by 100 using % ---] AS ""
     MEMBER ['0.0,' -->  1100] AS 1100, FORMAT_STRING =  '0.0,'
     MEMBER ['0.0,,' --> 1100000] AS 1100000, FORMAT_STRING =  '0.0,,'
     MEMBER ['0.0%%' --> 0.11] AS 0.11, FORMAT_STRING =  '0.0%%'
     MEMBER ['0.0,%%' --> 0.11] AS 0.11, FORMAT_STRING =  '0.0,%%'
 SELECT
   [Measures].allmembers -[Measures].members - [Answer] on 0
 FROM sales
 CELL PROPERTIES formatted_value
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">--- NUMERIC ---</td>
  <td class="ic3-mdx-result-cell-header">'#####.##'   --> 8315.4</td>
  <td class="ic3-mdx-result-cell-header">'##,###.##'  --> 8315.4</td>
  <td class="ic3-mdx-result-cell-header">'00000.00'   --> 8315.4</td>
  <td class="ic3-mdx-result-cell-header">'#####0.0##' --> 8315.4</td>
  <td class="ic3-mdx-result-cell-header">'$0'         --> 8315.4</td>
  <td class="ic3-mdx-result-cell-header">--- PERCENTAGE ---</td>
  <td class="ic3-mdx-result-cell-header">'0%'   --> 0.4231</td>
  <td class="ic3-mdx-result-cell-header">'0.0%' --> 0.4231</td>
  <td class="ic3-mdx-result-cell-header">--- SCIENTIFIC ---</td>
  <td class="ic3-mdx-result-cell-header">'0.00E+00' --> 42.31</td>
  <td class="ic3-mdx-result-cell-header">--- ROUNDING ---</td>
  <td class="ic3-mdx-result-cell-header">'0.0' -->  5.149</td>
  <td class="ic3-mdx-result-cell-header">'0.0' -->  5.150</td>
  <td class="ic3-mdx-result-cell-header">'0.0' --> -5.149</td>
  <td class="ic3-mdx-result-cell-header">'0.0' --> -5.150</td>
  <td class="ic3-mdx-result-cell-header">--- SECTIONS ---</td>
  <td class="ic3-mdx-result-cell-header">'0.#;Neg (0.#);0.00' --> +8315.115</td>
  <td class="ic3-mdx-result-cell-header">'0.#;Neg (0.#);0.00' --> -8315.115</td>
  <td class="ic3-mdx-result-cell-header">'0.#;Neg (0.#);0.00' --> 0.0</td>
  <td class="ic3-mdx-result-cell-header">--- Divide by 1000 with , and multiply by 100 using % ---</td>
  <td class="ic3-mdx-result-cell-header">'0.0,' -->  1100</td>
  <td class="ic3-mdx-result-cell-header">'0.0,,' --> 1100000</td>
  <td class="ic3-mdx-result-cell-header">'0.0%%' --> 0.11</td>
  <td class="ic3-mdx-result-cell-header">'0.0,%%' --> 0.11</td>
</tr>
<tr>
 <td></td>
 <td>8315.4</td>
 <td>8,315.4</td>
 <td>08315.40</td>
 <td>8315.4</td>
 <td>$8315</td>
 <td></td>
 <td>42%</td>
 <td>42.3%</td>
 <td></td>
 <td>4.23E+01</td>
 <td></td>
 <td>5.1</td>
 <td>5.2</td>
 <td>-5.1</td>
 <td>-5.2</td>
 <td></td>
 <td>8315.1</td>
 <td>Neg (8315.1)</td>
 <td>0.00</td>
 <td></td>
 <td>1.1</td>
 <td>1.1</td>
 <td>1100.0%%</td>
 <td>1.1%%</td>
</tr>
</table>



**Assertion : MDX Equals**

```
WITH
    MEMBER [--- NUMERIC ---] AS ""
    MEMBER ['#####.##'   --> 8315.4] AS "8315.4"
    MEMBER ['##,###.##'  --> 8315.4] AS "8,315.4"
    MEMBER ['00000.00'   --> 8315.4] AS  "08315.40"
    MEMBER ['#####0.0##' --> 8315.4] AS "8315.4"
    MEMBER ['$0'         --> 8315.4] AS "$8315"
    MEMBER [--- PERCENTAGE ---] AS ""
    MEMBER ['0%'   --> 0.4231] AS  "42%"
    MEMBER ['0.0%' --> 0.4231] AS  "42.3%"
    MEMBER [--- SCIENTIFIC ---] AS ""
    MEMBER ['0.00E+00' --> 42.31] AS  "4.23E+01"
    MEMBER [--- ROUNDING ---] AS ""
    MEMBER ['0.0' -->  5.149] AS "5.1"
    MEMBER ['0.0' -->  5.150] AS "5.2"
    MEMBER ['0.0' --> -5.149] AS "-5.1"
    MEMBER ['0.0' --> -5.150] AS "-5.2"
    MEMBER [--- SECTIONS ---] AS ""
    MEMBER ['0.#;Neg (0.#);0.00' --> +8315.115] AS "8315.1"
    MEMBER ['0.#;Neg (0.#);0.00' --> -8315.115] AS "Neg (8315.1)"
    MEMBER ['0.#;Neg (0.#);0.00' --> 0.0] AS "0.00"
    MEMBER [--- Divide by 1000 with , and multiply by 100 using % ---] AS ""
    MEMBER ['0.0,' -->  1100] AS "1.1"
    MEMBER ['0.0,,' --> 1100000] AS "1.1"
    MEMBER ['0.0%%' --> 0.11] AS "1100.0%%"
    MEMBER ['0.0,%%' --> 0.11] AS "1.1%%"
SELECT
  [Measures].allmembers -[Measures].members - [Answer] on 0
FROM sales
CELL PROPERTIES formatted_value
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Excel/VB : Date Formatting

</div>

<div class="ic3-mdx-example-body" style="display:none">

Some examples of date formatting following MDX standard notation.

**Query**

```
WITH
        MEMBER [MyDate] AS CDate('6/24/2001 16:23:45') // for SSAS compatibility, better use StringToDate('6/24/2001 16:23:45','M/d/yyyy HH:mm:ss')
        MEMBER ["m/d/yy"] AS [MyDate], FORMAT_STRING="m/d/yy"
        MEMBER ["dddd, mmmm dd"] AS [MyDate], FORMAT_STRING="dddd, mmmm dd"
        MEMBER ["d-mmm"] AS [MyDate], FORMAT_STRING="d-mmm"
        MEMBER ["mmmm-yy"] AS [MyDate], FORMAT_STRING="mmmm-yy"
        MEMBER ["hh:mm AM/PM"] AS [MyDate], FORMAT_STRING="hh:mm AM/PM"
        MEMBER ["h:mm:ss a/p"] AS [MyDate], FORMAT_STRING="h:mm:ss a/p"
        MEMBER ["d-mmmm h:mm"] AS [MyDate], FORMAT_STRING="d-mmmm h:mm"
        MEMBER ["d-mmmm-yyyy"] AS [MyDate], FORMAT_STRING="d-mmmm-yyyy"
        MEMBER ["d mmmm"] AS [MyDate], FORMAT_STRING="d mmmm"
        MEMBER ["mmmm yy"] AS [MyDate], FORMAT_STRING="mmmm yy"
        MEMBER ["h:mm"] AS [MyDate], FORMAT_STRING="h:mm"
        MEMBER ["h:mm:ss"] AS [MyDate], FORMAT_STRING="h:mm:ss"
        MEMBER ["m/d/yy h:mm"] AS [MyDate], FORMAT_STRING="m/d/yy h:mm"
SELECT
  [Measures].allmembers -[Measures].members - [MyDate] - [Answer] on 0
FROM sales
CELL PROPERTIES formatted_value
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">"m/d/yy"</td>
  <td class="ic3-mdx-result-cell-header">"dddd, mmmm dd"</td>
  <td class="ic3-mdx-result-cell-header">"d-mmm"</td>
  <td class="ic3-mdx-result-cell-header">"mmmm-yy"</td>
  <td class="ic3-mdx-result-cell-header">"hh:mm AM/PM"</td>
  <td class="ic3-mdx-result-cell-header">"h:mm:ss a/p"</td>
  <td class="ic3-mdx-result-cell-header">"d-mmmm h:mm"</td>
  <td class="ic3-mdx-result-cell-header">"d-mmmm-yyyy"</td>
  <td class="ic3-mdx-result-cell-header">"d mmmm"</td>
  <td class="ic3-mdx-result-cell-header">"mmmm yy"</td>
  <td class="ic3-mdx-result-cell-header">"h:mm"</td>
  <td class="ic3-mdx-result-cell-header">"h:mm:ss"</td>
  <td class="ic3-mdx-result-cell-header">"m/d/yy h:mm"</td>
</tr>
<tr>
 <td>6/24/01</td>
 <td>Sunday, June 24</td>
 <td>24-Jun</td>
 <td>June-01</td>
 <td>04:23 PM</td>
 <td>4:23:45 PM</td>
 <td>24-June 16:23</td>
 <td>24-June-2001</td>
 <td>24 June</td>
 <td>June 01</td>
 <td>16:23</td>
 <td>16:23:45</td>
 <td>6/24/01 16:23</td>
</tr>
</table>



**Assertion : MDX Equals**

```
WITH
        //  24 June 2001 16:23
        MEMBER ["m/d/yy"] AS "6/24/01"
        MEMBER ["dddd, mmmm dd"] AS "Sunday, June 24"
        MEMBER ["d-mmm"] AS  "24-Jun"
        MEMBER ["mmmm-yy"] AS "June-01"
        MEMBER ["hh:mm AM/PM"] AS "04:23 PM"
        MEMBER ["h:mm:ss a/p"] AS "4:23:45 PM"
        MEMBER ["d-mmmm h:mm"] AS "24-June 16:23"
        MEMBER ["d-mmmm-yyyy"] AS "24-June-2001"
        MEMBER ["d mmmm"] AS "24 June"
        MEMBER ["mmmm yy"] AS "June 01"
        MEMBER ["h:mm"] AS "16:23"
        MEMBER ["h:mm:ss"] AS "16:23:45"
        MEMBER ["m/d/yy h:mm"] AS "6/24/01 16:23"
SELECT
  [Measures].allmembers -[Measures].members - [Answer] on 0
FROM sales
CELL PROPERTIES formatted_value
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Java : Numeric

</div>

<div class="ic3-mdx-example-body" style="display:none">

Format a value using a JAVA format as described [here](https://docs.oracle.com/en/java/javase/25/docs/api/java.base/java/util/Formatter.html).

**Query**

```
with member val_ as 1000 / 3, FORMAT_STRING = 'j:%3.1f'
select from sales where val_ cell properties formatted_value
```

 
**Result**

<pre>333.3</pre>



**Assertion : MDX Equals**

```
with member val_ as "333.3"
select from sales where val_ cell properties formatted_value
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Java : Thousand Separator

</div>

<div class="ic3-mdx-example-body" style="display:none">

To display a (localized) thousand separator, use the comma character inside the JAVA format as following:

**Query**

```
with member val_ as 1234.56, FORMAT_STRING = 'j:%,.2f'
select from sales where val_ cell properties formatted_value
```

 
**Result**

<pre>1,234.56</pre>



**Assertion : MDX Equals**

```
with member val_ as "1,234.56"
select from sales where val_ cell properties formatted_value
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Java : Percent

</div>

<div class="ic3-mdx-example-body" style="display:none">

To format a value that represents a percentage, you'll have to escape the % character as demonstrated in this example. Note that the 'Percent' FORMAT_STRING will multiply
the value by 100.

**Query**

```
with member val_ as 50.0, FORMAT_STRING = 'j:%2.0f%%'
select from sales where val_ cell properties formatted_value
```

 
**Result**

<pre>50%</pre>



**Assertion : MDX Equals**

```
with member val_ as "50%"
select from sales where val_ cell properties formatted_value
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### Java : Date

</div>

<div class="ic3-mdx-example-body" style="display:none">

Format a date value using a JAVA simple date format as described [here](https://docs.oracle.com/en/java/javase/25/docs/api/java.base/java/text/SimpleDateFormat.html).

**Query**

```
with member val_ as now(), FORMAT_STRING = 'j:d-M-yyyy'
select from sales where val_ cell properties formatted_value
```

 
**Result**

<pre>21-11-2025</pre>



**Assertion : MDX Equals**

```
with
  member currentTime as now()
  member val_ as str(day(currentTime)) + "-" + str(month(currentTime)) + "-" + str(year(currentTime))
select from sales where val_ cell properties formatted_value
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### formatter('duration', 'pattern')

</div>

<div class="ic3-mdx-example-body" style="display:none">

The function formatter('duration', 'pattern') allows to format a duration value expressed in milliseconds. Available pattern tags are:
<ul>
    <li> %y : year  (no leap)
    <li> %M : month (as 30 days)
    <li> %d : day
    <li> %h : hour
    <li> %m : minute
    <li> %s : second
    <li> %S : millisecond
</ul>

Note that in the following example, the leading number of hours is greater than 1 day and the trailing seconds are not displayed.

**Query**

```
with member val_ as (25 * 3600 * 1000) + 25 * 60 * 1000 + 30 * 1000, FORMAT_STRING = formatter("duration", "%hh %mm")
select from sales where val_ cell properties formatted_value
```

 
**Result**

<pre>25h 25m</pre>



**Assertion : MDX Equals**

```
with member val_ as "25h 25m"
select from sales where val_ cell properties formatted_value
```


 
</div>

           

<div class="ic3-mdx-example-header ic3-mdx-example-header-collapsed" onclick="ic3_mdx_example_click(this)">

#### MDX expression (Java format)

</div>

<div class="ic3-mdx-example-body" style="display:none">

As for any other CELL PROPERTIES an MDX expression can be used to define the FORMAT_STRING. In this example, we are demonstrating how to test for Nan/Infinite numbers and display a dedicated message accordingly. Note the usage of a declared FUNCTION; a better solution would be to define this function once at schema level and re-use it in many places.

**Query**

```
with
  function myFormatString() as
             iif( isNaN( currentCellValue() ),
                  'j:invalid number',
                  iif( isInfinity( currentCellValue() ), 'j:infinite number', 'j:%f' )
             )

  member nan as 0.0 / 0.0, FORMAT_STRING = myFormatString()
  member infinity as 1.0 / 0.0, FORMAT_STRING = myFormatString()

select { nan, infinity } on 0 from sales cell properties formatted_value
```

 
**Result**

<table  class="ic3-mdx-result">
<tr>
  <td class="ic3-mdx-result-cell-header">nan</td>
  <td class="ic3-mdx-result-cell-header">infinity</td>
</tr>
<tr>
 <td>invalid number</td>
 <td>infinite number</td>
</tr>
</table>



**Assertion : MDX Equals**

```
with
  member nan as 0.0 / 0.0, FORMAT_STRING = 'j:invalid number'
  member infinity as 1.0 / 0.0, FORMAT_STRING = 'j:infinite number'

select { nan, infinity } on 0 from sales cell properties formatted_value
```


 
</div>

                                                                                                            

### See Also


[Cell Properties](../Cell%20Properties.md)

[LANGUAGE](LANGUAGE.md)


### XMLA Properties

Find [here](../icCubeDefaultXmlaProperties.md) the value of the XMLA properties when not specified otherwise.

_
