/*
 * Copyright 1999 - 2016 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.plugin.olapfunctions.timedimension;

import crazydev.iccube.builder.type.OlapBuilderInputType;
import crazydev.iccube.collection.OlapIterator;
import crazydev.iccube.enums.OlapLevelType;
import crazydev.iccube.exception.OlapErrorCode;
import crazydev.iccube.olap.component.naming.OlapNameContext;
import crazydev.iccube.olap.entity.OlapEntity;
import crazydev.iccube.olap.entity.level.OlapLevel;
import crazydev.iccube.olap.entity.member.OlapMember;
import crazydev.iccube.olap.entity.scalar.OlapNonScalarEntity;
import crazydev.iccube.olap.entity.scalar.OlapNumericEntity;
import crazydev.iccube.olap.entity.set.OlapSetFactory;
import crazydev.iccube.olap.entity.set.OlapTupleSet;
import crazydev.iccube.olap.eval.exception.OlapFunctionEvaluationException;
import crazydev.iccube.olap.eval.execinstr.gf.context.GFContext;
import crazydev.iccube.olap.eval.execinstr.gf.function.GFFunctionArgs;
import crazydev.iccube.olap.eval.filter.dimension.OlapMemberFilter;
import crazydev.iccube.olap.eval.function.OlapFunctionArgs;
import crazydev.iccube.olap.eval.function.OlapFunctionArgumentType;
import crazydev.iccube.olap.eval.function.mdx.member.OlapMdxTimeNavigationFunction;
import org.jetbrains.annotations.Nullable;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;

import java.util.ArrayList;
import java.util.Arrays;

/**
 * icCube function to navigate in the time dimension
 */
public abstract class OlapTimeDimensionFunction extends OlapMdxTimeNavigationFunction
{
    protected static final OlapFunctionArgs ARGS_1 = new OlapFunctionArgs(1);

    protected static final OlapFunctionArgs ARGS_2 = new OlapFunctionArgs(2)
    {
        @Override
        public OlapFunctionArgumentType getType(int pos)
        {
            return pos < 1 ? super.getType(pos) : OlapFunctionArgumentType.VALUE;
        }
    };

    protected static final OlapFunctionArgs ARGS_3 = new OlapFunctionArgs(3)
    {
        @Override
        public OlapFunctionArgumentType getType(int pos)
        {
            return pos < 1 ? super.getType(pos) : OlapFunctionArgumentType.VALUE;
        }
    };

    static final String PREFIX = "dt";

    OlapTimeDimensionFunction(String name, OlapFunctionArgs args)
    {
        super(PREFIX + name, args);
    }

    OlapTimeDimensionFunction(String name)
    {
        this(name, ARGS_1);
    }

    @Override
    public OlapNonScalarEntity evalForCluster(GFContext context, GFFunctionArgs args)
    {
        final OlapEntity arg = args.getNonMissingArgEntity(context, 0, "member|set");
        if (arg.isMdxNull())
        {
            return asNull();
        }

        final OlapMemberFilter memberFilter = args.getSchemaPermissionsWithoutPerspective(context).getDimensionsPermissions(args.getCube(context)).getMemberFilter();
        if (arg.isToMemberApplicable(context.getPrepareContext()))
        {
            final OlapMember member = args.toMember(context, 0, true, true);
            return asNullable(getNavigatedMember(context, args, member, memberFilter));
        }
        else
        {
            final OlapTupleSet<OlapMember> set = args.toMemberSet(context, 0, false);
            final OlapIterator<? extends OlapMember> iter = set.getIterator(false);
            OlapMember member;
            iter.reset();
            ArrayList<OlapMember> newSet = new ArrayList<>();
            while ((member = iter.next()) != null)
            {
                final OlapMember navigatedMember = getNavigatedMember(context, args, member, memberFilter);
                if (navigatedMember != null)
                {
                    newSet.add(navigatedMember);
                }
            }
            return newSet.isEmpty() ? OlapSetFactory.empty() : OlapSetFactory.instance(newSet);
        }
    }

    @Nullable
    private OlapMember getNavigatedMember(GFContext context, GFFunctionArgs args, @Nullable OlapMember member, @Nullable OlapMemberFilter memberFilter)
    {
        if (member == null || member.isMdxNull())
        {
            return null;
        }
        if (member.isAll())
        {
            return member;
        }

        // evaluate parameters
        OlapNumericEntity[] parameters = null;
        if (args.getArgLength() > 1)
        {
            parameters = new OlapNumericEntity[args.getArgLength()];
            for (int i = 1; i < args.getArgLength(); i++)
            {
                parameters[i] = args.toNumeric(context, i);
                final String error = validateParameter(i, parameters[i]);
                if (error != null)
                {
                    throw new OlapFunctionEvaluationException(args.getErrorContext(context, i), getName(), OlapErrorCode.FUNCTION_ARGUMENT_ERROR, i, error);
                }
            }
        }

        final OlapLevel level = member.getLevel();
        final OlapLevelType levelType = level.getLevelType();
        final OlapBuilderInputType[] keyType = level.getMemberKeyType();
        if (keyType != null && (keyType.length != 1 || !keyType[0].isDateType()))
        {
            return args.onFunctionError(context, OlapErrorCode.LEVEL_DATETIME_KEYTYPE_EXPECTED, level.getUniqueName(OlapNameContext.DEFAULT_VALUES), Arrays.toString(keyType));
        }

        final OlapLevelType levelChanged = getLevelChanged();
        if (levelType != null && levelChanged != null && (levelType.isBefore(levelChanged)))
        {
            return args.onFunctionError(context, OlapErrorCode.LEVEL_DATETIME_EXPECTED_LEVEL_TYPE, level.getUniqueName(OlapNameContext.DEFAULT_VALUES), levelChanged.toString(), Arrays.toString(keyType));
        }

        final Comparable transformedDateTime;
        try
        {
            final Comparable dateTime = member.getKeyValue();
            if (dateTime instanceof LocalDate)
            {
                LocalDate date = (LocalDate) dateTime;
                LocalDate t = apply(date, parameters);
                if (level.getLevelType() == OlapLevelType.WEEK)
                {
                    t = t.withDayOfWeek(1);
                }
                transformedDateTime = t;
            }
            else if (dateTime instanceof LocalDateTime)
            {
                LocalDateTime time = (LocalDateTime) dateTime;
                LocalDateTime t = apply(time, parameters);
                if (level.getLevelType() == OlapLevelType.WEEK)
                {
                    t = t.withDayOfWeek(1);
                }
                transformedDateTime = t;
            }
            else
            {
                return args.onFunctionError(context, OlapErrorCode.LEVEL_DATETIME_KEYTYPE_EXPECTED, level.getUniqueName(OlapNameContext.DEFAULT_VALUES), dateTime.getClass().getSimpleName());
            }
        }
        catch (IllegalArgumentException ex)
        {
            // Joda safeguard as it should be catched before, this is a wrong parameter on the month, year... it might be the second parameter..
            throw new OlapFunctionEvaluationException(args.getErrorContext(context, 1), getName(), OlapErrorCode.FUNCTION_ARGUMENT_ERROR, 1, ex.toString());
        }

        final OlapMember retMember = level.lookupMemberByKeyValue(transformedDateTime);
        return retMember == null || (memberFilter != null && !memberFilter.isAccepting(retMember)) ? null : retMember;
    }

    @Nullable
    protected String validateParameter(int i, OlapNumericEntity parameter)
    {
        return null;
    }

    int firstAsInteger(OlapNumericEntity[] parameters)
    {
        return (int) parameters[1].value();
    }

    int secAsInteger(OlapNumericEntity[] parameters)
    {
        return (int) parameters[2].value();
    }

    protected abstract LocalDateTime apply(LocalDateTime time, OlapNumericEntity[] parameters);

    protected abstract LocalDate apply(LocalDate time, OlapNumericEntity[] parameters);

    @Nullable
    protected abstract OlapLevelType getLevelChanged();

}
