/*
 * Copyright 2014 - 2018 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.plugin.olapfunctions.timedimension;

import crazydev.iccube.enums.OlapLevelType;
import crazydev.iccube.exception.OlapErrorCode;
import crazydev.iccube.olap.component.naming.OlapNameContext;
import crazydev.iccube.olap.entity.OlapEntity;
import crazydev.iccube.olap.entity.dimension.OlapDimension;
import crazydev.iccube.olap.entity.hierarchy.OlapHierarchy;
import crazydev.iccube.olap.entity.level.OlapLevel;
import crazydev.iccube.olap.entity.member.OlapMember;
import crazydev.iccube.olap.entity.scalar.OlapNonScalarEntity;
import crazydev.iccube.olap.eval.execinstr.gf.context.GFContext;
import crazydev.iccube.olap.eval.execinstr.gf.function.GFFunctionArgs;
import crazydev.iccube.olap.eval.filter.dimension.OlapMemberFilter;
import crazydev.iccube.olap.eval.function.OlapFunctionArgs;
import crazydev.iccube.olap.eval.function.mdx.member.OlapMdxTimeNavigationFunction;
import org.jetbrains.annotations.Nullable;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.LocalTime;

import java.util.Arrays;

abstract class OlapTimeDimensionAsOfBase extends OlapMdxTimeNavigationFunction
{
    OlapTimeDimensionAsOfBase(String name, OlapFunctionArgs args)
    {
        super(name, args);
    }

    @Override
    public OlapNonScalarEntity evalForCluster(GFContext context, GFFunctionArgs args)
    {
        final OlapEntity arg0 = args.getNonMissingArgEntity(context, 0, "hierarchy|level");
        if (arg0.isMdxNull())
        {
            return asNull();
        }

        final OlapLevel level;
        if (arg0 instanceof OlapLevel)
        {
            level = (OlapLevel) arg0;
        }
        else
        {
            if (arg0 instanceof OlapDimension)
            {
                level = args.toOnlyHierarchy(context, 0, (OlapDimension) arg0).getLastLevel();
            }
            else if (arg0 instanceof OlapHierarchy)
            {
                level = ((OlapHierarchy) arg0).getLastLevel();
            }
            else
            {
                return args.onFunctionArgTypeMismatchError(context, 0, "hierarchy|dimension|level", arg0.getFriendlyTypeName());
            }
        }

        final boolean isDateType;
        if (level.isKeyType(LocalDate.class))
        {
            isDateType = true;
        }
        else if (level.isKeyType(LocalDateTime.class))
        {
            isDateType = false;
        }
        else
        {
            return args.onFunctionError(context, OlapErrorCode.LEVEL_DATETIME_KEYTYPE_EXPECTED, level.getUniqueName(OlapNameContext.DEFAULT_VALUES), Arrays.toString(level.getMemberKeyType()));
        }

        final Object transformedDateTime = getLookupKey(context, args, isDateType);
        if (transformedDateTime == null)
        {
            return asNull();
        }

        OlapMember retMember = level.lookupMemberByKeyValue(transformedDateTime);
        final OlapLevelType levelType = level.getLevelType();
        if (retMember == null && levelType != null)
        {
            if (transformedDateTime instanceof LocalDateTime dateTime)
            {
                final LocalDateTime t = levelType.transformToValidDate(dateTime, context.getLocale());
                retMember = level.lookupMemberByKeyValue(t);
            }
            else if (transformedDateTime instanceof LocalDate dateTime)
            {
                final LocalDateTime t = levelType.transformToValidDate(dateTime.toLocalDateTime(new LocalTime(0)), context.getLocale());
                retMember = level.lookupMemberByKeyValue(t.toLocalDate());
            }
        }

        final OlapMemberFilter memberFilter = args.getSchemaPermissionsWithoutPerspective(context).getDimensionsPermissions(args.getCube(context)).getMemberFilter();
        return retMember == null || (memberFilter != null && !memberFilter.isAccepting(retMember)) ? asNull() : retMember;
    }

    @Nullable
    abstract Object getLookupKey(GFContext context, GFFunctionArgs args, boolean isDateType);
}
