/*
 * Copyright 2014 - 2017 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.plugin.olapfunctions.timedimension;

import crazydev.common.exception.programming.CdShouldNotBeHereProgrammingException;
import crazydev.iccube.builder.type.OlapBuilderInputType;
import crazydev.iccube.collection.olapiterator.OlapCompactIterator;
import crazydev.iccube.enums.OlapLevelType;
import crazydev.iccube.exception.OlapErrorCode;
import crazydev.iccube.olap.component.naming.OlapNameContext;
import crazydev.iccube.olap.entity.level.OlapLevel;
import crazydev.iccube.olap.entity.member.OlapMember;
import crazydev.iccube.olap.entity.scalar.OlapNonScalarEntity;
import crazydev.iccube.olap.entity.set.OlapSetFactory;
import crazydev.iccube.olap.entity.set.OlapTupleSet;
import crazydev.iccube.olap.eval.execinstr.gf.context.GFContext;
import crazydev.iccube.olap.eval.execinstr.gf.function.GFFunctionArgs;
import crazydev.iccube.olap.eval.function.OlapFunctionArgs;
import crazydev.iccube.olap.eval.function.mdx.member.OlapMdxTimeNavigationFunction;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;

import java.util.Arrays;
import java.util.function.Predicate;

/**
 * icCube function to navigate in the time dimension
 */
public abstract class OlapDatesTimeDimensionFunction extends OlapMdxTimeNavigationFunction
{
    protected static final OlapFunctionArgs ARGS_1 = new OlapFunctionArgs(1, 2);

    private OlapDatesTimeDimensionFunction(String name, OlapFunctionArgs args)
    {
        super(OlapTimeDimensionFunction.PREFIX + name, args);
    }

    OlapDatesTimeDimensionFunction(String name)
    {
        this(name, ARGS_1);
    }

    @Override
    public OlapNonScalarEntity evalForCluster(GFContext context, GFFunctionArgs args)
    {
        final OlapMember member = args.toMember(context, 0, true, true);
        if (member == null || member.isMdxNull())
        {
            return OlapSetFactory.empty();
        }

        final boolean compact = args.toOptionalBoolean(context, 1, compactDefaultValue());

        final OlapLevel level = member.getLevel();
        final OlapLevelType levelType = level.getLevelType();
        if (level.isAll() || (levelType != null && (levelType.isBefore(getLevelChanged()))))
        {
            if (errorOnLevelMismatch())
            {
                return args.onFunctionError(context, OlapErrorCode.LEVEL_DATETIME_EXPECTED_LEVEL_TYPE, level.getUniqueName(OlapNameContext.DEFAULT_VALUES), "DAY", levelType == null ? "ALL" : levelType.toString());
            }
            else
            {
                return OlapSetFactory.empty();
            }
        }
        final OlapBuilderInputType[] keyType = level.getMemberKeyType();
        if (keyType != null && (keyType.length != 1 || !keyType[0].isDateType()))
        {
            return args.onFunctionError(context, OlapErrorCode.LEVEL_DATETIME_KEYTYPE_EXPECTED, level.getUniqueName(OlapNameContext.DEFAULT_VALUES), Arrays.toString(keyType));
        }

        final Comparable key = member.getKeyValue();
        final Predicate<OlapMember> filter;
        if (key instanceof LocalDate)
        {
            LocalDate date = (LocalDate) key;
            filter = filter(date);
        }
        else if (key instanceof LocalDateTime)
        {
            LocalDateTime time = (LocalDateTime) key;
            filter = filter(time);
        }
        else
        {
            return args.onFunctionError(context, OlapErrorCode.LEVEL_DATETIME_KEYTYPE_EXPECTED, level.getUniqueName(OlapNameContext.DEFAULT_VALUES), key.getClass().getSimpleName());
        }

        final OlapMember endStartPeriodMember = getEndPeriodMember(member);
        if (endStartPeriodMember == null)
        {
            return OlapSetFactory.empty();
        }

        // the algo goes backwards by default
        final OlapTupleSet<OlapMember> memberSet = endStartPeriodMember.navigateMembers(args.getMemberFilter(context), filter, direction());
        return compact ? OlapCompactIterator.newInstance(memberSet, false) : memberSet;
    }

    protected boolean errorOnLevelMismatch()
    {
        return false;
    }

    @Nullable
    protected OlapMember getEndPeriodMember(OlapMember member)
    {
        return member;
    }

    protected abstract Predicate<OlapMember> filter(LocalDateTime time);

    protected abstract Predicate<OlapMember> filter(LocalDate time);

    protected boolean compactDefaultValue()
    {
        return true;
    }

    protected boolean direction()
    {
        return false;
    }

    protected OlapLevelType getLevelChanged()
    {
        return OlapLevelType.DAY;
    }

    @NotNull
    protected OlapMember getLastExistingWeekday(OlapMember member, int firstWeekDay, int lastWeekDay)
    {
        final Comparable key = member.getKeyValue();

        OlapMember nmember;
        if (key instanceof LocalDate)
        {
            LocalDate date = (LocalDate) key;
            for (int i = lastWeekDay; i >= firstWeekDay; i--)
            {
                nmember = member.getLevel().lookupMemberByKeyValue(date.withDayOfWeek(i));
                if (nmember != null)
                {
                    return nmember;
                }
            }
        }
        else if (key instanceof LocalDateTime)
        {
            LocalDateTime time = (LocalDateTime) key;
            for (int i = lastWeekDay; i >= firstWeekDay; i--)
            {
                nmember = member.getLevel().lookupMemberByKeyValue(time.withDayOfWeek(i));
                if (nmember != null)
                {
                    return nmember;
                }
            }
        }

        throw new CdShouldNotBeHereProgrammingException();
    }

}
