/*
 * Copyright 2014 - 2020 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.plugin.olapfunctions.timedimension;

import crazydev.iccube.collection.olapiterator.OlapCompactIterator;
import crazydev.iccube.enums.OlapLevelType;
import crazydev.iccube.exception.OlapErrorCode;
import crazydev.iccube.olap.component.naming.OlapNameContext;
import crazydev.iccube.olap.entity.level.OlapLevel;
import crazydev.iccube.olap.entity.member.OlapMember;
import crazydev.iccube.olap.entity.scalar.OlapNonScalarEntity;
import crazydev.iccube.olap.entity.set.OlapSetFactory;
import crazydev.iccube.olap.entity.set.OlapTupleSet;
import crazydev.iccube.olap.eval.execinstr.gf.context.GFContext;
import crazydev.iccube.olap.eval.execinstr.gf.function.GFFunctionArgs;
import crazydev.iccube.olap.eval.function.OlapFunctionArgs;
import crazydev.iccube.olap.eval.function.mdx.member.OlapMdxTimeNavigationFunction;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;

import java.util.function.Predicate;

/**
 * icCube function to navigate in the time dimension
 */
public class OlapDatesRollingMonthsTimeDimensionFunction extends OlapMdxTimeNavigationFunction
{
    public static final String NAME = OlapTimeDimensionFunction.PREFIX + "RollingMonths";

    protected static final OlapFunctionArgs ARGS = new OlapFunctionArgs(2, 3);

    public OlapDatesRollingMonthsTimeDimensionFunction()
    {
        super(NAME, ARGS);
    }

    @Override
    public OlapNonScalarEntity evalForCluster(GFContext context, GFFunctionArgs args)
    {
        final OlapMember endPeriod = args.toMember(context, 0, true, true);
        if (endPeriod == null || endPeriod.isMdxNull() || endPeriod.isAll())
        {
            return OlapSetFactory.empty();
        }
        final OlapLevel level = endPeriod.getLevel();

        final int rollingMonths = args.toInteger(context, 1);
        final boolean compact = args.toOptionalBoolean(context, 2, true);

        final OlapLevelType levelType = level.getLevelType();
        if (levelType == null)
        {
            return args.onFunctionError(context, OlapErrorCode.LEVEL_TYPE_TIME_EXPECTED, level.getUniqueName(OlapNameContext.DEFAULT_VALUES), "NULL");
        }
        switch (levelType)
        {
            case YEAR:
                if (rollingMonths % 12 != 0)
                {
                    return onFunctionArgTypeMismatchError(context, 1, "value that is a multiplier of 12", "" + rollingMonths, " (it's a Year level, you can't start in the middle)");
                }
                break;
            case HALF_YEAR:
                if (rollingMonths % 6 != 0)
                {
                    return onFunctionArgTypeMismatchError(context, 1, "value that is a multiplier of 6", "" + rollingMonths, " (it's a Half-Year level, you can't start in the middle)");
                }
                break;
            case QUARTER:
                if (rollingMonths % 3 != 0)
                {
                    return onFunctionArgTypeMismatchError(context, 1, "value that is a multiplier of 3", "" + rollingMonths, " (it's a Quarter level, you can't be start the middle)");
                }
                break;
            case MONTH:
            case WEEK:
            case DAY:
            case HOUR:
            case HALF_HOUR:
                break;
            case DAY_MONTH:
            case DAY_YEAR:
            default:
            {
                return args.onFunctionError(context, OlapErrorCode.LEVEL_TYPE_TIME_EXPECTED, level.getUniqueName(OlapNameContext.DEFAULT_VALUES), levelType.getName());
            }
        }

        final Comparable key = endPeriod.getKeyValue();
        final boolean direction = rollingMonths < 0;  /*  switch start / end */

        final Predicate<OlapMember> filter;
        if (key instanceof LocalDate)
        {
            LocalDate startPeriodMemberDate = ((LocalDate) key).minusMonths(rollingMonths);
            filter = olapMember -> {
                final LocalDate currentDate = (LocalDate) olapMember.getKeyValue();
                return direction ? currentDate.isBefore(startPeriodMemberDate) /* actually end */ : currentDate.isAfter(startPeriodMemberDate);
            };
        }
        else if (key instanceof LocalDateTime)
        {
            LocalDateTime startPeriodMemberDate = ((LocalDateTime) key).minusMonths(rollingMonths);
            filter = olapMember -> {
                final LocalDateTime currentDate = (LocalDateTime) olapMember.getKeyValue();
                return direction ? currentDate.isBefore(startPeriodMemberDate) /* actually end */ : currentDate.isAfter(startPeriodMemberDate);
            };
        }
        else
        {
            return args.onFunctionError(context, OlapErrorCode.LEVEL_DATETIME_KEYTYPE_EXPECTED, level.getUniqueName(OlapNameContext.DEFAULT_VALUES), key.getClass().getSimpleName());
        }

        final OlapTupleSet<OlapMember> memberSet = endPeriod.navigateMembers(
                args.getMemberFilter(context),
                filter,
                direction);

        return compact ? OlapCompactIterator.newInstance(memberSet, false) : memberSet;
    }
}
