/*
 * Copyright 1999 - 2016 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.plugin.olapfunctions.others;

import crazydev.iccube.olap.entity.OlapEmptyEntity;
import crazydev.iccube.olap.entity.OlapEntity;
import crazydev.iccube.olap.entity.scalar.OlapNumericEntity;
import crazydev.iccube.olap.entity.scalar.OlapScalarEntity;
import crazydev.iccube.olap.entity.set.OlapTupleSet;
import crazydev.iccube.olap.entity.tuple.OlapTuple;
import crazydev.iccube.olap.eval.execinstr.gf.context.GFContext;
import crazydev.iccube.olap.eval.execinstr.gf.function.GFFunctionArgs;
import crazydev.iccube.olap.eval.function.OlapFunctionArgs;
import crazydev.iccube.olap.eval.function.mdx.numeric.OlapBaseGenericStatisticalFunction;
import crazydev.iccube.olap.eval.function.mdx.numeric.calculators.OlapSetCalculatorOneGenExpression;
import crazydev.iccube.olap.eval.operator.scalar.OlapScalarOperators;
import org.jetbrains.annotations.Nullable;

public class OlapProdFunction extends OlapBaseGenericStatisticalFunction
{
    public static final String NAME = "Prod";

    public static final OlapFunctionArgs ARGS = new OlapFunctionArgs(1, 3)
    {
        @Override
        public boolean isLambda(int pos)
        {
            return pos == 1;
        }
    };

    public OlapProdFunction()
    {
        super(NAME, ARGS);
    }

    @Override
    protected OlapSetCalculatorOneGenExpression createCalculator(GFFunctionArgs args, OlapTupleSet set)
    {
        final boolean includeEmpty = getIncludeEmpty(args);

        return new OlapSetCalculatorOneGenExpression()
        {
            @Nullable
            OlapScalarEntity prodValue;

            /**
             * If not -1 means the prodValue has been evaluated to 0 and the iteration/processing can be stopped.
             */
            int stopIterationOrdinal = -1;

            @Override
            public boolean isResultEmpty()
            {
                return prodValue == null || prodValue.isMdxNull();
            }

            @Override
            public OlapEntity calculate()
            {
                return prodValue == null ? asEmpty() : prodValue;
            }

            @Override
            public int stopIterationOrdinal()
            {
                return stopIterationOrdinal;
            }

            @Override
            public void addNull(OlapTuple setTuple, int iTupleOrdinal)
            {
                if (stopIterationOrdinal != -1 /* zero already */)
                {
                    return;
                }

                if (includeEmpty)
                {
                    // keep empty till it's empty
                    if (prodValue == null || prodValue.isMdxNull())
                    {
                        prodValue = OlapEmptyEntity.INSTANCE;
                    }
                    else
                    {
                        prodValue = prodValue.multiplyByZero();

                        if (stopIterationOrdinal != -1)
                        {
                            stopIterationOrdinal = 0 /* can stop as we've reached 0 */;
                        }
                    }
                }
            }

            @Override
            public void addValue(GFContext context, OlapTuple setTuple, int iTupleOrdinal, OlapScalarEntity value)
            {
                if (stopIterationOrdinal != -1 /* zero already */)
                {
                    return;
                }

                // the new value is zero, always zero
                if (value.isZero())
                {
                    prodValue = value;

                    if (stopIterationOrdinal != -1)
                    {
                        stopIterationOrdinal = 0 /* can stop as we've reached 0 */;
                    }

                    return;
                }

                // we've added an empty value (includeempty is true), so always zero
                if (prodValue == OlapEmptyEntity.INSTANCE)
                {
                    prodValue = value.multiplyByZero();

                    if (stopIterationOrdinal != -1)
                    {
                        stopIterationOrdinal = 0 /* can stop as we've reached 0 */;
                    }

                    return;
                }

                if (prodValue == null)
                {
                    if (value instanceof OlapNumericEntity)
                    {
                        OlapNumericEntity numericEntity = (OlapNumericEntity) value;
                        prodValue = new OlapNumericEntity(numericEntity.doubleValue());
                    }
                    else
                    {
                        prodValue = value.asConstant();
                    }
                }
                else
                {
                    prodValue = OlapScalarOperators.multiply(prodValue, value);
                }
            }
        };
    }

}