/*
 * Copyright 2014 - 2019 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.plugin.olapfunctions.others;

import crazydev.iccube.cluster.shared.schema.S_NonEmptyBehavior;
import crazydev.iccube.cluster.shared.schema.S_NonEmptyBehaviorFactoryContext;
import crazydev.iccube.cluster.shared.schema.S_NonEmptyBehaviorOperator;
import crazydev.iccube.collection.OlapIterator;
import crazydev.iccube.collection.olapiterator.OlapFilterIterator;
import crazydev.iccube.collection.olapiterator.fastcj.OlapFastCrossjoinIterator;
import crazydev.iccube.exception.OlapErrorCode;
import crazydev.iccube.olap.entity.member.OlapMember;
import crazydev.iccube.olap.entity.scalar.OlapNonScalarEntity;
import crazydev.iccube.olap.entity.scalar.OlapNumericEntity;
import crazydev.iccube.olap.entity.set.OlapIteratorTupleSet;
import crazydev.iccube.olap.entity.set.OlapSetFactory;
import crazydev.iccube.olap.entity.set.OlapTupleSet;
import crazydev.iccube.olap.entity.tuple.OlapTuple;
import crazydev.iccube.olap.entity.tuple.dimensionality.OlapTupleDimensionality;
import crazydev.iccube.olap.eval.exception.OlapEvaluationException;
import crazydev.iccube.olap.eval.execinstr.gf.context.GFContext;
import crazydev.iccube.olap.eval.execinstr.gf.function.GFFunctionArgs;
import crazydev.iccube.olap.eval.function.OlapFunctionArgs;
import crazydev.iccube.olap.eval.function.OlapFunctionArgumentType;
import crazydev.iccube.olap.eval.function.OlapNonScalarEntityFunction;
import crazydev.iccube.olap.eval.select.context.OlapFastCrossjoinContext;
import crazydev.iccube.olap.eval.select.context.OlapNonEmptyBehaviorException;
import crazydev.iccube.olap.loggers.OlapLoggers;
import org.jetbrains.annotations.Nullable;

public class OlapNonEmptyXFunction extends OlapNonScalarEntityFunction
{
    public static final String NAME = "NonEmptyX";

    public static final OlapFunctionArgs ARGS = new OlapFunctionArgs(2, 3)
    {
        @Override
        public OlapFunctionArgumentType getType(int pos)
        {
            if (pos == 2)
            {
                return OlapFunctionArgumentType.VALUE;
            }
            return super.getType(pos);
        }
    };

    public OlapNonEmptyXFunction()
    {
        this(NAME);
    }

    public OlapNonEmptyXFunction(String name)
    {
        super(name, ARGS);
    }

    @Override
    public OlapNonScalarEntity evalForCluster(GFContext context, GFFunctionArgs args)
    {
        final OlapTupleSet<? extends OlapTuple> set = args.toSet(context, 0);

        if (!(set instanceof OlapIteratorTupleSet))
        {
            return args.onFunctionArgTypeMismatchError(context, 0, "crossjoin", set.getFriendlyTypeName());
        }

        final OlapIteratorTupleSet iteratorSet = (OlapIteratorTupleSet) set;

        iteratorSet.hackForNonEmptyX() /* bit lost => newInstance later... */;

        OlapIterator tupleIterator = iteratorSet.getIterator(false);
        OlapFilterIterator filter = null;

        if (tupleIterator instanceof OlapFilterIterator)
        {
            filter = (OlapFilterIterator) tupleIterator;
            tupleIterator = filter.getUnderlyingIterator();
        }

        if (!(tupleIterator instanceof OlapFastCrossjoinIterator))
        {
            return args.onFunctionArgTypeMismatchError(context, 0, "crossjoin", set.getFriendlyTypeName());
        }

        final OlapFastCrossjoinIterator fastCrossjoinIterator = (OlapFastCrossjoinIterator) tupleIterator;

        final OlapTupleDimensionality dimensionality = fastCrossjoinIterator.getTupleDimensionality();

        final OlapMember member = args.toMember(context, 1);

        if (!member.isMeasure() || member.isCalculated())
        {
            // We're not supporting calc. measure (member) for now as NonEmptyX does not ensure to return non-empty
            // tuples: indeed the calculated non-empty-behavior is a best effort so NonEmptyX should perform an actual
            // non-empty processing at the end...

            return args.onFunctionArgTypeMismatchError(context, 1, "measure", member.getFriendlyTypeName());
        }

        final OlapNumericEntity limit = args.isMissingArg(2) ? null : args.toNumeric(context, 2);

        // final GFEvalTuple stackTuple = context.setupEvalTuple(
        //         true, false, OlapBlankTuple.INSTANCE
        // );

        final S_NonEmptyBehavior nonEmptyBehaviors = getNEB(context, dimensionality, member);

        if (nonEmptyBehaviors != null && nonEmptyBehaviors.hasAggregationWithRollupHierarchy(OlapLoggers.MDX_EVALUATION_FAST_CROSSJOIN, "[fast-crossjoin] non-empty-x : "))
        {
            OlapLoggers.MDX_EVALUATION.info("[non-empty-x] aggregation w/ rollup hierarchy not supported");

            throw new OlapEvaluationException(context, OlapErrorCode.SELECT_EVAL_UNEXPECTED_ERROR, "non-empty-x : fast crossjoin not supported for aggregation w/ rollup hierarchy");
        }

        final OlapFastCrossjoinContext cjContext = new OlapFastCrossjoinContext(
                context.getStatementExecutionContext(), context, nonEmptyBehaviors
        );

        OlapIterator<OlapTuple> crossjoin = fastCrossjoinIterator.fastCrossjoinForNonEmptyProcessing(cjContext);

        crossjoin.hackForNonEmptyX();

        if (crossjoin == null)
        {
            throw new OlapEvaluationException(context, OlapErrorCode.SELECT_EVAL_UNEXPECTED_ERROR, "non-empty-x : fast crossjoin not enabled");
        }

        if (limit != null && !limit.isEmpty())
        {
            final int value = limit.intValue();
            crossjoin = crossjoin.head(value);
        }

        if (filter != null)
        {
            // reset the filter (mainly where clause)
            return OlapSetFactory.instance(filter.withUnderlying(crossjoin));
        }
        else
        {
            return OlapSetFactory.instance(crossjoin).asTupleList();
        }

    }

    private S_NonEmptyBehavior getNEB(GFContext context, @Nullable OlapTupleDimensionality dimensionality, OlapTuple tuple)
    {
        if (dimensionality == null /* dunno */)
        {
            OlapLoggers.MDX_EVALUATION_FAST_CROSSJOIN.info("[fast-crossjoin] non-empty-x : null dimensionality => reverting to non-empty-behavior = $one$");

            return S_NonEmptyBehavior.one() /* bad luck */;
        }

        try
        {
            final S_NonEmptyBehaviorFactoryContext nonEmptyBehaviorContext = context.createNonEmptyBehaviorContext(
                    dimensionality
            );

            final S_NonEmptyBehavior nonEmptyBehaviors = S_NonEmptyBehaviorOperator.tuple(
                    nonEmptyBehaviorContext, tuple
            );

            return nonEmptyBehaviors;

        }
        catch (OlapNonEmptyBehaviorException ex)
        {
            OlapLoggers.MDX_EVALUATION_FAST_CROSSJOIN.error("[fast-crossjoin] non-empty-x : non-empty-behavior error", ex);

            throw new OlapEvaluationException(context, OlapErrorCode.SELECT_EVAL_UNEXPECTED_ERROR, "non-empty-x error : " + ex.getMessage());
        }
    }
}
