/*
 * Copyright 1999 - 2014 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.plugin.olapfunctions.others;

import crazydev.common.mdx.CdMdxColorUtils;
import crazydev.common.utils.CdTimeUtils;
import crazydev.iccube.olap.compiler.OlapCompilationContext;
import crazydev.iccube.olap.eval.function.OlapFunctionCallInstr;
import crazydev.iccube.olap.eval.instr.OlapInstr;
import crazydev.iccube.olap.eval.instr.OlapInstrLocationRange;
import crazydev.iccube.olap.eval.literal.OlapLocalDateLiteralInstr;
import crazydev.iccube.olap.eval.literal.OlapLocalDateTimeLiteralInstr;
import crazydev.iccube.olap.eval.literal.OlapLongLiteralInstr;
import crazydev.iccube.olap.loggers.OlapLoggers;
import crazydev.iccube.plugin.function.OlapPluginEvaluationException;
import crazydev.iccube.plugin.function.OlapPluginUserJavaFunction;
import org.jetbrains.annotations.Nullable;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;

@SuppressWarnings({"UnusedDeclaration"})
public abstract class OlapIcCubeFunctions extends OlapPluginUserJavaFunction
{
// see OlapPatternMatchesFunction
//
//    public static final OlapIcCubeFunctions PATTERNMATCHES = new OlapIcCubeFunctions(2, String.class, String.class)
//    {
//        @Override
//        protected Object doEval(Object[] args)
//        {
//            String s0 = (String) args[0];
//            String s1 = (String) args[1];
//            return s0 == null || s1 == null ? false : s1.matches(s0);
//        }
//    };

    public static final OlapIcCubeFunctions CONTAINS = new OlapIcCubeFunctions(2, String.class, String.class)
    {
        @Override
        protected Object doEval(Object[] args)
        {
            String s0 = (String) args[0];
            String s1 = (String) args[1];
            return s0 == null || s1 == null ? false : s0.contains(s1);
        }
    };

    public static final OlapPluginUserJavaFunction ISNAN = new SingleArgDoubleFunction()
    {
        @Override
        protected Boolean evaluate(double d)
        {
            return Double.isNaN(d);
        }
    };

    /* ---------------------------------------------------------------------- */

    public static final OlapPluginUserJavaFunction ISINFINITY = new SingleArgDoubleFunction()
    {
        @Override
        protected Boolean evaluate(double d)
        {
            return Double.isInfinite(d);
        }
    };

    /* ---------------------------------------------------------------------- */

    public static final OlapPluginUserJavaFunction TODAY = new OlapPluginUserJavaFunction(0, 0)
    {
        @Override
        protected Object doEval(Object[] args)
        {
            final LocalDate ic3 = CdTimeUtils.redefinedToday();

            if(ic3 != null)
            {
                return ic3;
            }

            return LocalDate.now();
        }
    };

    public static final OlapPluginUserJavaFunction DATETIME = new OlapIcCubeFunctions(2, 6, Integer.class, Integer.class, Integer.class, Integer.class, Integer.class, Integer.class)
    {
        @Override
        @Nullable
        public OlapInstr asInstr(OlapCompilationContext context, OlapInstrLocationRange location, OlapFunctionCallInstr.Notation mdxNotation, OlapInstr[] args)
        {
            try
            {
                if (args.length == 6)
                {
                    final OlapInstr year = args[0];
                    final OlapInstr month = args[1];
                    final OlapInstr day = args[2];

                    final OlapInstr hours = args[3];
                    final OlapInstr minutes = args[4];
                    final OlapInstr seconds = args[5];

                    if (
                            year instanceof OlapLongLiteralInstr
                            && month instanceof OlapLongLiteralInstr
                            && day instanceof OlapLongLiteralInstr
                            && hours instanceof OlapLongLiteralInstr
                            && (minutes == null || minutes instanceof OlapLongLiteralInstr)
                            && (seconds == null || seconds instanceof OlapLongLiteralInstr)
                    )
                    {
                        return new OlapLocalDateTimeLiteralInstr(location, new LocalDateTime(
                                asYear(year), asNumeric(month), asNumeric(day),
                                asNumeric(hours), asNumeric(minutes), asNumeric(seconds)
                        ));
                    }

                }

                if (args.length == 5)
                {
                    final OlapInstr year = args[0];
                    final OlapInstr month = args[1];
                    final OlapInstr day = args[2];

                    final OlapInstr hours = args[3];
                    final OlapInstr minutes = args[4];

                    if (
                            year instanceof OlapLongLiteralInstr
                            && month instanceof OlapLongLiteralInstr
                            && day instanceof OlapLongLiteralInstr
                            && hours instanceof OlapLongLiteralInstr
                            && (minutes == null || minutes instanceof OlapLongLiteralInstr)
                    )
                    {
                        return new OlapLocalDateTimeLiteralInstr(location, new LocalDateTime(
                                asYear(year), asNumeric(month), asNumeric(day),
                                asNumeric(hours), asNumeric(minutes)
                        ));
                    }

                }

                if (args.length == 3)
                {
                    final OlapInstr year = args[0];
                    final OlapInstr month = args[1];
                    final OlapInstr day = args[2];

                    if (
                            year instanceof OlapLongLiteralInstr
                            && month instanceof OlapLongLiteralInstr
                            && (day == null || day instanceof OlapLongLiteralInstr)
                    )
                    {
                        return new OlapLocalDateLiteralInstr(location, new LocalDate(
                                asYear(year), asNumeric(month), asNumeric(day)
                        ));
                    }

                }
            }
            catch (RuntimeException ignored)
            {
                // dunno but it seems the error is not properly reported...
                OlapLoggers.MDX_EVALUATION.warn("[mdx] could not optimize DateTime()", ignored);
            }

            return super.asInstr(context, location, mdxNotation, args);
        }

        private int asYear(OlapInstr<?> value)
        {
            final int year = ((OlapLongLiteralInstr) value).getValue().intValue();
            return year + (year > 100 ? 0 : 2000);
        }

        private int asNumeric(@Nullable OlapInstr<?> value)
        {
            if (value == null)
            {
                return 0;
            }
            return ((OlapLongLiteralInstr) value).getValue().intValue();
        }

        @Override
        protected Object doEval(Object[] args) throws OlapPluginEvaluationException
        {
            checkNotNull(args);

            int year = (Integer) args[0];
            final int month = (Integer) args[1];
            final int day = args.length > 2 ? (Integer) args[2] : 0;

            year = year + (year > 100 ? 0 : 2000);
            if (args.length <= 3)
            {
                return new LocalDate(year, month, day);
            }

            final int hours = (Integer) args[3];
            final int minutes = args.length > 4 ? (Integer) args[4] : 0;
            final int seconds = args.length > 5 ? (Integer) args[5] : 0;

            return new LocalDateTime(year, month, day, hours, minutes, seconds);
        }
    };

    public static final OlapPluginUserJavaFunction ToHexColor = new OlapPluginUserJavaFunction(1, int.class)
    {
        @Override
        protected Object doEval(Object[] args)
        {
            int decimal = (int) args[0];
            return CdMdxColorUtils.hexValue(decimal);
        }
    };

    protected OlapIcCubeFunctions(int minArgs, Class... parametersType)
    {
        super(minArgs, parametersType);
    }

    protected OlapIcCubeFunctions(int minArgs, int maxArgs, Class... parametersType)
    {
        super(minArgs, maxArgs, parametersType);
    }

    protected OlapIcCubeFunctions(Class[]... params)
    {
        super(params);
    }

    private static void checkNotNull(Object[] args) throws OlapPluginEvaluationException
    {
        for (int i = 0; i < args.length; i++)
        {
            if (args[i] == null)
            {
                throw new OlapPluginEvaluationException(OlapPluginEvaluationException.Kind.VALUE_INVALID, "parameter " + (i + 1) + " can not be null");
            }
        }
    }

//    private static abstract class SingleArgTextFunction extends OlapIcCubeFunctions
//    {
//        protected SingleArgTextFunction()
//        {
//            super(1, String.class);
//        }
//
//        @Override
//        protected Object doEval(Object[] args)
//        {
//            return evaluate((String) args[0]);
//        }
//
//        protected abstract Object evaluate(String arg);
//    }

    private static abstract class SingleArgDoubleFunction extends OlapIcCubeFunctions
    {
        protected SingleArgDoubleFunction()
        {
            super(1, Double.class);
        }

        @Override
        protected Object doEval(Object[] args)
        {
            final Double arg = (Double) args[0];
            return arg == null ? isNull() : evaluate(arg);
        }

        protected abstract Object evaluate(double arg);

        public boolean isNull()
        {
            return false;
        }
    }
}
