/*
 * Copyright 1999 - 2014 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.plugin.olapfunctions.excel;

import crazydev.iccube.plugin.function.OlapPluginEvaluationException;
import crazydev.iccube.plugin.function.OlapPluginUserJavaFunction;
import org.joda.time.CdJodaTimeUtil;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

/**
 * http://office.microsoft.com/client/helppreview14.aspx?AssetId=HV080007662&lcid=1033&NS=EXCEL%2EDEV&Version=14&tl=2
 */
@SuppressWarnings({"UnusedDeclaration"})
public class OlapVBExcelConversionFunctions
{
    public static abstract class OneArg extends OlapPluginUserJavaFunction
    {
        protected OneArg()
        {
            super(1, Object.class);
        }

        @Override
        public final Object doEval(Object[] args) throws OlapPluginEvaluationException
        {
            return evaluate(args[0]);
        }

        protected abstract Object evaluate(Object obj) throws OlapPluginEvaluationException;
    }

    // Excel TimeLine ( later: https://msdn.microsoft.com/en-us/library/az4se3k1(v=vs.110).aspx )

    private static final SimpleDateFormat DATE_FORMATS[] = {
            new SimpleDateFormat("yyyy-MM-dd", Locale.US),
            new SimpleDateFormat("M/d/y h:m:s", Locale.US),
            new SimpleDateFormat("M/d/y h:m:s", Locale.US),
            new SimpleDateFormat("M/d/y h:m", Locale.US),
            new SimpleDateFormat("M/d/y", Locale.US),
            new SimpleDateFormat("M-d-y h:m:s", Locale.US),
            new SimpleDateFormat("M-d-y h:m", Locale.US),
            new SimpleDateFormat("M-d-y", Locale.US),
            new SimpleDateFormat("h:m:s", Locale.US),
            new SimpleDateFormat("h:m", Locale.US)
    };

    public static final OlapPluginUserJavaFunction CDate = new OlapPluginUserJavaFunction(1, 1, String.class)
    {
        @Override
        protected Object doEval(Object[] args) throws OlapPluginEvaluationException
        {
            final String dateAsString = (String) (args[0]);

            if (dateAsString == null)
            {
                return null /* SSAS */;
            }

            final int pos = dateAsString.indexOf("-");
            final boolean useUniversal = pos == 4;

            synchronized (DATE_FORMATS)
            {
                for (int idx = 0; idx < DATE_FORMATS.length; idx++)
                {
                    if (idx == 0 && !useUniversal)
                    {
                        continue;
                    }

                    final SimpleDateFormat format = DATE_FORMATS[idx];

                    try
                    {
                        final Date date = format.parse(dateAsString);
                        return new LocalDateTime(date);
                    }
                    catch (ParseException ex)
                    {
                        // ignore
                    }
                }
            }
            throw new RuntimeException("CDate: unable to parse date [" + dateAsString + "]");
        }
    };

    public static final OlapPluginUserJavaFunction CDbl = new OneArg()
    {
        @Override
        protected Object evaluate(Object d) throws OlapPluginEvaluationException
        {
            if (d == null)
            {
                return null /* SSAS */;
            }

            if (d instanceof String)
            {
                return Double.valueOf((String) d);
            }

            if (d instanceof Float)
            {
                return ((Float) d).doubleValue();
            }

            if (d instanceof Double)
            {
                return d;
            }

            if (d instanceof Long)
            {
                return ((Long) d).doubleValue();
            }

            if (d instanceof Integer)
            {
                return ((Integer) d).doubleValue();
            }

            if (d instanceof Short)
            {
                return ((Short) d).doubleValue();
            }

            if (d instanceof Byte)
            {
                return ((Byte) d).doubleValue();
            }

            throw new RuntimeException("CDbl: invalid argument type [" + d + "]");
        }

    };

    public static final OlapPluginUserJavaFunction CLng = new OneArg()
    {
        @Override
        protected Object evaluate(Object d) throws OlapPluginEvaluationException
        {
            if (d == null)
            {
                return null /* SSAS */;
            }

            if (d instanceof String)
            {
                return Integer.valueOf((String) d);
            }

            if (d instanceof Float)
            {
                return ((Double) Math.rint((Float) d)).intValue();
            }

            if (d instanceof Double)
            {
                return ((Double) Math.rint((Double) d)).intValue();
            }

            if (d instanceof Long)
            {
                return ((Long) d).intValue();
            }

            if (d instanceof Integer)
            {
                return d;
            }

            if (d instanceof Short)
            {
                return ((Short) d).intValue();
            }

            if (d instanceof Byte)
            {
                return ((Byte) d).intValue();
            }

            throw new RuntimeException("CLng: invalid argument type [" + d + "]");
        }

    };

    public static final OlapPluginUserJavaFunction CLngLng = new OneArg()
    {
        @Override
        protected Object evaluate(Object d) throws OlapPluginEvaluationException
        {
            if (d == null)
            {
                return null /* SSAS */;
            }

            if (d instanceof String)
            {
                return Long.valueOf((String) d);
            }

            if (d instanceof Float)
            {
                return ((Double) Math.rint((Float) d)).longValue();
            }

            if (d instanceof Double)
            {
                return ((Double) Math.rint((Double) d)).longValue();
            }

            if (d instanceof Long)
            {
                return d;
            }

            if (d instanceof Integer)
            {
                return ((Integer) d).longValue();
            }

            if (d instanceof Short)
            {
                return ((Short) d).longValue();
            }

            if (d instanceof Byte)
            {
                return ((Byte) d).longValue();
            }

            throw new RuntimeException("CLngLng: invalid argument type [" + d + "]");
        }
    };

    public static final OlapPluginUserJavaFunction CSng = new OneArg()
    {
        @Override
        protected Object evaluate(Object d) throws OlapPluginEvaluationException
        {
            if (d == null)
            {
                return null /* SSAS */;
            }

            if (d instanceof String)
            {
                return Float.valueOf((String) d);
            }

            if (d instanceof Float)
            {
                return d;
            }

            if (d instanceof Double)
            {
                return ((Double) d).floatValue();
            }

            if (d instanceof Long)
            {
                return ((Long) d).floatValue();
            }

            if (d instanceof Integer)
            {
                return ((Integer) d).floatValue();
            }

            if (d instanceof Short)
            {
                return ((Short) d).floatValue();
            }

            if (d instanceof Byte)
            {
                return ((Byte) d).floatValue();
            }

            throw new RuntimeException("CSng: invalid argument type [" + d + "]");
        }

    };

    public static final OlapPluginUserJavaFunction CStr = new OneArg()
    {
        @Override
        protected Object evaluate(Object d) throws OlapPluginEvaluationException
        {
            if (d == null)
            {
                return null /* SSAS */;
            }

            if (d instanceof String)
            {
                return d;
            }

            if (d instanceof Boolean)
            {
                return ((Boolean) d) ? "True" : "False" /* MS */;
            }

            if (d instanceof Float)
            {
                return d.toString();
            }

            if (d instanceof Double)
            {
                return d.toString();
            }

            if (d instanceof Long)
            {
                return d.toString();
            }

            if (d instanceof Integer)
            {
                return d.toString();
            }

            if (d instanceof Short)
            {
                return d.toString();
            }

            if (d instanceof Byte)
            {
                return d.toString();
            }

            if (d instanceof Date)
            {
                return d.toString();
            }

            if (d instanceof LocalDateTime)
            {
                return d.toString();
            }

            if (d instanceof LocalDate)
            {
                return d.toString();
            }

            throw new RuntimeException("CStr: invalid argument type [" + d + "]");
        }
    };

    public static final OlapPluginUserJavaFunction Val = new OneArg()
    {
        @Override
        protected Object evaluate(Object d) throws OlapPluginEvaluationException
        {
            if (d == null)
            {
                return 0 /* SSAS */;
            }

            if (d instanceof String)
            {
                return evaluateForString((String) d);
            }

            if (d instanceof Boolean)
            {
                return ((Boolean) d) ? -1 : 0;
            }

            if (d instanceof Float)
            {
                return d;
            }

            if (d instanceof Double)
            {
                return d;
            }

            if (d instanceof Long)
            {
                return d;
            }

            if (d instanceof Integer)
            {
                return d;
            }

            if (d instanceof Short)
            {
                return d;
            }

            if (d instanceof Byte)
            {
                return d;
            }

            if (d instanceof Date /* MS seems to return underlying value : e.g., Now() */)
            {
                return ((Date) d).getTime();
            }

            if (d instanceof LocalDateTime /* MS seems to return underlying value : e.g., Now() */)
            {
                return CdJodaTimeUtil.toDefaultChronologyMillis((LocalDateTime) d);
            }

            if (d instanceof LocalDate /* MS seems to return some sort of underlying value : e.g., Now() */)
            {
                return CdJodaTimeUtil.toDefaultChronologyDays((LocalDate) d);
            }

            throw new RuntimeException("Val: invalid argument type [" + d + "]");
        }

        private Object evaluateForString(String str)
        {
            final ValStringEvaluator evaluator = new ValStringEvaluator(str);
            return evaluator.evaluate() /* not thread-safe */;
        }

    };

    /**
     * Not thread-safe as using fields ...
     */
    public static final class ValStringEvaluator
    {
        private final String str;

        private final StringBuilder sb = new StringBuilder();

        private int pos = 0;

        public ValStringEvaluator(String str)
        {
            this.str = str;
        }

        public Object evaluate()
        {
            if (str.isEmpty())
            {
                return 0;
            }

            // +-  0-9 . 0-9 e  +- 0-9

            eatWhiteSpaces();
            scanOptionalSign();

            final boolean hasDigit = scanDigits() /* eating white spaces ... */;

            final boolean hasDot = scanOptionalDot();

            if (!hasDigit && !hasDot)
            {
                return 0;
            }

            if (hasDot)
            {
                scanDigits() /* eating white spaces ... */;
            }

            final boolean hasExponentMarker = scanOptionalExponentMarker();

            if (hasExponentMarker)
            {
                final int exponentMarkerPos = pos - 1;

                eatWhiteSpaces();
                scanOptionalSign();

                final boolean hasExponent = scanDigits() /* eating white spaces ... */;

                if (!hasExponent /* unread : e +- */)
                {
                    final String val = sb.toString();

                    sb.delete(0, val.length());
                    sb.append(val.substring(0, exponentMarkerPos));
                }
            }

            // Time to parse the value ...

            final String valueAsString = sb.toString();

            if (valueAsString.length() == 0)
            {
                return 0;
            }

            // MS seems to returns always integer values when possible ...

            final double value = Double.parseDouble(valueAsString);

            if (value > Long.MAX_VALUE || value < Long.MIN_VALUE)
            {
                return value;
            }

            final long valueAsLong = (long) value;

            if (valueAsLong == value)
            {
                return valueAsLong;
            }

            return value;
        }

        private void eatWhiteSpaces()
        {
            while (pos < str.length() && Character.isSpaceChar(str.charAt(pos)))
            {
                ++pos;
            }
        }

        private void scanOptionalSign()
        {
            if (pos < str.length())
            {
                char cc = str.charAt(pos);

                if ('+' == cc || '-' == cc)
                {
                    sb.append(cc);
                    ++pos;
                }
            }
        }

        private boolean scanDigits()
        {
            boolean hasDigit = false;

            while (pos < str.length())
            {
                final char cc = str.charAt(pos);

                if (cc >= '0' && cc <= '9')
                {
                    hasDigit = true;

                    sb.append(cc);
                    ++pos;
                }
                else if (Character.isSpaceChar(cc))
                {
                    ++pos;
                }
                else
                {
                    break;
                }
            }

            return hasDigit;
        }

        private boolean scanOptionalDot()
        {
            if (pos < str.length())
            {
                char cc = str.charAt(pos);

                if ('.' == cc)
                {
                    sb.append(cc);
                    ++pos;

                    return true;
                }
            }

            return false;
        }

        private boolean scanOptionalExponentMarker()
        {
            if (pos < str.length())
            {
                char cc = str.charAt(pos);

                if ('e' == cc || 'E' == cc)
                {
                    sb.append(cc);
                    ++pos;

                    return true;
                }
            }

            return false;
        }

    }

}
