## Vector (MDX++)

icCube MDX support for vector objects allows to perform fundamental operations of numerical linear algebra.
Several ways of constructing vectors are available via the MDX functions:

* [Vector()](../mdx/Vector.md)
* [VectorA()](../mdx/VectorA.md)
* [VectorK()](../mdx/VectorK.md)
* [VectorN()](../mdx/VectorN.md)
* [VectorS()](../mdx/VectorS.md)

### Operators

The MDX operators have been extended to support basic arithmetic operations (i.e., addition, multiplication,
division, power) and comparison (i.e., equality, comparison). Some basic examples :

    VectorN(2,3) + 2 = VectorN(4,5)
    VectorN(2,3) - 2 = VectorN(0,1)
    VectorN(2,3) * 2 = VectorN(4,6)
    VectorN(2,3) / 2 = VectorN(1,3/2)
    VectorN(2,3) * VectorN(1,2) = VectorN(2*1,3*2)
    VectorN(2,3) ^ VectorN(1,2) = 2*1 + 3*2 = 8       -- the 'scalar or dot product'

### Numerical Vector - Object-Oriented Methods

The notation `vector->method( ... )` allows for accessing the methods of the vector. The conversion of the
parameters (as well as the result) is handled automatically by the MDX engine. As an example, the following code
returns the arithmetic mean of the amount of sales for all the countries:

    Vector( [Geography].[Geo].[Country].members, [Measures].[Amount] ) -> mean()

#### aggregate()

Returns the sum of the values in the vector using icCube's default algorithm (same as the one used in the measures)

#### copy()

Returns a copy of the vector.

#### contains(Double value)

Returns true if the value is present in the vector.

#### corr(Vector b)

Returns a scalar value with the Pearson correlation between the two vectors.

#### cumSum()

Returns a scalar with the cumulative sum: `cumSum = a + a+b + a+b+c -> {a,b,c,...}`.

#### divide(Double value)

Returns a new vector where the specified value is divided to each element of the original vector.

#### first()

Returns the first element of the vector (`Double.NaN` if empty vector).

#### fft()

Returns a vector with the Fourier fast-forward transformation of the vector (absolute value of the complex value).

#### geometricMean()

Returns the geometric mean (`exp( 1/n (sum of logs) )`) of the elements of the vector.

#### get(double n)

Returns the `n-th` element of the vector. If value is a numeric smaller than `1.0` returns the `n * (length - 1)`
element.

#### head(double size)

Returns a vector with the first `size` elements. If value is less than `1.0` return the `size *( length() + 1)`
first elements.

#### hist(double start, double stop)

Returns the number of elements of the vector between `start` (inclusive) and `stop` (exclusive). A null value
for `start` means `-inf` whereas a null value for `stop` means `+inf`.

#### indexOf(Double value)

Returns the position in the vector of the values, -1 if not found

#### last()

Returns the last element of the vector (`Double.NaN` if empty vector).

#### length()

Returns the number of elements of the vector.

#### max()

Returns the maximum of the elements of the vector.

#### mean()

Returns the arithmetic mean of the elements of the vector.

#### median()

Returns the median of the elements of the vector. This is the same as `percentile(50.0)`.

#### min()

Returns the minimum of the elements of the vector.

#### minus(Double value)

Returns a new vector where the specified value is subtracted to each element of the original vector.

#### multiply(Double value)

Returns a new vector where the specified value is multiplied to each element of the original vector.

#### norm()

Returns the Euclidean norm (i.e., the square root of the sum of the squares).

#### round(int precision)

Returns a vector with all values rounded to the specified number of decimal places (values are rounded using
the `ROUND_HALF_UP` method).

#### percentile(double p)

Returns an estimate of the `p-th` percentile of the values in the vector (`100.0` equals `100%`).

#### pAbs()

Returns a vector with for each position the absolute value: `Vi = Abs( Ai )`.

#### pDif(Vector b)

Returns a vector with for each position the absolute value of the difference: `Vi = Abs( Ai - Bi )`.

#### phist(double start, double stop)

Returns `hist(start,stop) / length`.

#### pMax(Vector b)

Returns a vector with for each position the maximum of both vectors: `Vi = Max( Ai, Bi )`.

#### pMin(Vector b)

Returns a vector with for each position the minimum of both vectors: `Vi = Min( Ai, Bi )`.

#### parallelSort()

Returns a vector with the values sorted in ascending order using
[parallelSort](https://docs.oracle.com/javase/8/docs/api/java/util/Arrays.html#parallelSort-double:A-).

#### power(Double value)

Returns a new vector where the specified value is raised to the power to each element of the original vector.

#### product()

Returns the product of the values in the vector.

#### reverse()

Returns a vector with the order of the elements reversed (from last to first).

#### scale()

Returns a vector with all values scaled between `0.0` and `1.0`: `NewVi = Vi - Min(V) / Max(V) - Min(V)`

#### sumOn(OlapDoubleVectorEntity p)

Returns the sum of all values for each position defined by the vector:

    ForEach(Pi) { value += V->get(Pi); }    -- Pi positions outside bound are ignored

#### sort()

Returns a vector with the values sorted in ascending order.

#### stdev()

Returns the standard deviation of the values in the vector (square root of the variance).

#### sum()

Returns the sum of the values in the vector.

#### sumHP()

Returns the sum of the values in the vector using the high precision algorithm (i.e. Neumaier).

#### sum(Double value)

Returns a new vector where the specified value is added to each element of the original vector.

#### sumLog()

Returns the sum of the natural logs of the values in the vector.

#### sumSq()

Returns the sum of the squares of the values in the vector.

#### tail(double size)

Returns a vector with the last `size` elements. If value is less than `1.0` return the `size * ( length() + 1 )`
last elements.

#### value(index)

Returns the value of the `index` element of the vector.

#### variance()

Returns the variance of the values in the vector: `sum((x_i - mean)^2) / (n - 1)`.

### Set Operators

#### unique()

Returns the list sorted of unique values as a vector.

#### setUnion(OlapDoubleVectorEntity v2)

The union of two vector: `v ∪ v2`. Return a vector that is the result of adding all values of `v2` to the vector.
Use `unique()` if you don't want to have the same value twice.

#### setIntersect(OlapDoubleVectorEntity v2)

The intersection of two vectors: `v ∩ v2`. Returns a list that is the result of filtering out the values that are
not found in the vector `v2`. Use `unique()` if you don't want to have the same value twice, but remember that if
the vector `v` is unique the result will be unique too.

#### setMinus(OlapDoubleVectorEntity v2)

The difference between two vectors: `v \ v2`. Returns a list that is the result of filtering out the values that
are found in the vector `v`. Use `unique()` if you don't want to have the same value twice, but remember that if
the vector `v` is unique the result will be unique too.

### Comparable Vector - Object Oriented Methods

It is also possible to have vector on comparable values. For example using a Vector on a string column.

This vector can contain null values if an existing row has a null value (it is not the case for numerical values).

#### count()

Returns the size of the vector.

#### contains(value)

Returns `true` if the vector contains the value. Returns `false` otherwise.

#### filterNulls()

Returns the comparable vector without null values.

#### first()

Returns the first element of the vector (returns `null` if the vector is empty or if the first value is a `null`).

#### firstNotNull()

Returns the first not null element of the vector.

#### last()

Returns the last element of the vector (returns `null` if the vector is empty or if the last value is a `null`).

#### lastNotNull()

Returns the last not null element of the vector.

#### get(double n)

Returns the `n-th` element of the vector.

#### join(String delimiter)

Returns `join(delimiter, "", "")`. See below.

#### join(String delimiter, String prefix, String suffix)

Joins the values using the delimiter, then returns `<prefix>` + `<joined values>` + `<suffix>`. 

#### max()

Returns the maximum of the elements of the vector.

#### min()

Returns the minimum of the elements of the vector.

#### sameValues()

Returns `true` if all values are the same or if the size of the vector is 1 or 0. Returns `false` otherwise.

#### sort()

Returns a sorted comparable vector.

#### unique()

Returns a comparable vector with the unique values.

