/*
 * Copyright 2014 - 2020 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.googleapi.drive.datasource;

import crazydev.common.property.CdProperty;
import crazydev.common.property.CdReadWriteProperty;
import crazydev.iccube.builder.model.def.IOlapBuilderTabularDataDef;
import crazydev.iccube.builder.ux.meta.datasource.UxBuilderDataSourceType;
import crazydev.iccube.olap.component.context.OlapRuntimeContext;
import jakarta.xml.bind.annotation.XmlAttribute;
import jakarta.xml.bind.annotation.XmlRootElement;
import jakarta.xml.bind.annotation.XmlTransient;
import org.jetbrains.annotations.Nullable;

import java.util.List;

@XmlRootElement(name = "goSpreadsheetDS")
public class OlapBuilderGoogleSpreadsheetDataSource extends OlapBuilderGoogleDriveDataSource<OlapBuilderGoogleSpreadsheetConnection>
{
    public static final CdProperty FILE_NAME = new CdReadWriteProperty(OlapBuilderGoogleSpreadsheetDataSource.class, "fileName");

    public static final CdProperty FILE_ID = new CdReadWriteProperty(OlapBuilderGoogleSpreadsheetDataSource.class, "fileId", false);

    public static final CdProperty TIME_OUT = new CdReadWriteProperty(OlapBuilderGoogleSpreadsheetDataSource.class, "googleTimeOut", false);

    public static final CdProperty CACHE_FILE_SEC = new CdReadWriteProperty(OlapBuilderGoogleSpreadsheetDataSource.class, "cacheFileSec");

    public static final CdProperty WITHOUT_FORMULAS = new CdReadWriteProperty(OlapBuilderGoogleSpreadsheetDataSource.class, "excelWithoutFormulas");

    @XmlAttribute(required = true)
    public String fileName;

    @XmlAttribute(required = true)
    public String fileId;

    @XmlAttribute(required = false)
    public int cacheFileSec;

    @XmlAttribute(required = false)
    private boolean excelWithoutFormulas;

    @XmlTransient
    private OlapBuilderGoogleSpreadsheetConnection cachedConnection;

    @XmlTransient
    private long cacheMillis;

    @Nullable
    @XmlAttribute(required = false)
    private Integer googleTimeOut;

    public OlapBuilderGoogleSpreadsheetDataSource()
    {
    }

    public OlapBuilderGoogleSpreadsheetDataSource(String serviceAccountPrivateKey, String fileName)
    {
        super(serviceAccountPrivateKey);
        this.fileName = fileName;
    }

    public String getFileId()
    {
        return fileId;
    }

    public void setFileId(String fileId)
    {
        this.fileId = fileId;
    }

    public void setFileName(String fileName)
    {
        this.fileName = fileName;
    }

    public String getFileName()
    {
        return fileName;
    }

    @Override
    protected String getReportDataSourceType()
    {
        return "googleSpreadsheet";
    }

    @Override
    protected String getReportDataSourceTypeCaption()
    {
        return "Google Spreadsheet";
    }

    @Override
    public boolean isDiscoverTablesSupported()
    {
        return true;
    }

    /**
     * @return a possibly null list of table that can be created by this datasource.
     */
    @Override
    @Nullable
    public List<Class<? extends IOlapBuilderTabularDataDef>> getCreatedTableTypes()
    {
        return null /* cannot create any table */;
    }

    @Override
    public OlapBuilderGoogleSpreadsheetDataTable createEmptyDiscoverTable(String tableName)
    {
        return new OlapBuilderGoogleSpreadsheetDataTable(tableName);
    }

    @Override
    public OlapBuilderGoogleSpreadsheetConnection createConnection(OlapRuntimeContext context, boolean forceRefresh)
    {
        if (!forceRefresh && cachedConnection != null && (System.currentTimeMillis() - cacheMillis < cacheFileSec * 1000))
        {
            return cachedConnection;
        }
        OlapBuilderGoogleSpreadsheetConnection connection = new OlapBuilderGoogleSpreadsheetConnection(this, cacheFileSec > 0);
        if (cacheFileSec > 0)
        {
            cachedConnection = connection;
            cacheMillis = System.currentTimeMillis();
        }
        return connection;
    }

    @Override
    public List<String> discoverAllTablesNames(OlapBuilderGoogleSpreadsheetConnection connection, boolean filterSystemSchemas, final @Nullable String filter)
    {
        return connection.getSheetNames(sheetName -> acceptTableNameForDiscoverTableNames(filter, sheetName));
    }

    @Override
    @Nullable
    public Integer getTimeout()
    {
        return googleTimeOut;
    }

    public void setExcelWithoutFormulas(boolean excelWithoutFormulas)
    {
        this.excelWithoutFormulas = excelWithoutFormulas;
    }

    public boolean isWithoutFormulas()
    {
        return excelWithoutFormulas;
    }

    @Override
    protected UxBuilderDataSourceType createUxType()
    {
        return new UxBuilderDataSourceType(getTypeId(), getUxTypeImage(), () -> new OlapBuilderGoogleSpreadsheetDataSourceUxWizard(this, getTypeId()));
    }
}
