/*
 * Copyright 2014 - 2020 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.googleapi.common;

import crazydev.common.exception.programming.CdShouldNotBeHereProgrammingException;
import crazydev.common.property.CdProperty;
import crazydev.common.property.CdReadWriteProperty;
import crazydev.common.utils.CdStringUtils;
import crazydev.iccube.builder.errors.OlapBuilderErrorException;
import crazydev.iccube.builder.googleapi.errors.OlapGoogleApiError;
import crazydev.iccube.builder.model.def.IOlapBuilderDataSource;
import crazydev.iccube.builder.ux.meta.common.model.UxAutomodel;
import crazydev.iccube.builder.ux.meta.common.wizard.UxBuilderWizardStep;
import crazydev.iccube.builder.ux.meta.common.wizard.UxBuilderWizardValidation;
import crazydev.iccube.builder.ux.meta.datasource.plugin.UxBuilderGADataSourceAuthorizationType;
import crazydev.iccube.builder.ux.meta.datasource.wizard.UxBuilderCreateDataSourceWizardSteps;
import crazydev.iccube.builder.ux.meta.datasource.wizard.UxBuilderCreateTypedDataSourceWizard;
import crazydev.iccube.builder.ux.meta.datasource.wizard.UxBuilderDataSourceNameForm;
import crazydev.iccube.fs.OlapFile;
import crazydev.iccube.olap.component.context.OlapRuntimeContext;
import jakarta.xml.bind.annotation.XmlAttribute;
import org.jetbrains.annotations.NotNull;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import static crazydev.iccube.builder.googleapi.common.OlapBuilderGoogleApiDataSource.GOOGLE_OAUTH2_PROPERTY;

public abstract class OlapBuilderGoogleDataSourceUxWizard<DS extends OlapBuilderGoogleApiDataSource, WEBTOKEN_MODEL1, WEBTOKEN_MODEL2 extends OlapBuilderWizardWebTokenUxModel> extends UxBuilderCreateTypedDataSourceWizard
{
    protected final DS ds;

    public OlapBuilderGoogleDataSourceUxWizard(String dataSourceTypeId, List<UxBuilderWizardStep> steps, DS datasource)
    {
        super(dataSourceTypeId, steps);
        this.ds = datasource;
    }

    /**
     * Validate the wizard models until the 'activeStep' possibly returning new steps.
     *
     * @throws OlapBuilderErrorException properly handled in the UI
     */
    public UxBuilderWizardValidation validate(OlapRuntimeContext context, int activeStep)
    {
        int base = 0;

        if (isGoogleOAuthDisabled(context))
        {
            // skip the Authorization
            base++;
        }

        return onServiceAccountWizard(context, activeStep, base + 1);

    }

    private UxBuilderWizardValidation onServiceAccountWizard(OlapRuntimeContext context, int activeStep, int base)
    {
        if ((activeStep + base) == 1)
        {
            return onServiceAccount(context, activeStep);
        }
        if ((activeStep + base) == 2)
        {
            return onServiceAccountScope(context, activeStep);
        }
        return null;
    }

    /**
     * Restore the server side data source from edited models of the wizards.
     */
    @Override
    public IOlapBuilderDataSource getDataSourceFromUI()
    {
        final UxBuilderGADataSourceAuthorizationType authType = getStepModelFor(UxBuilderGADataSourceAuthorizationType.class);

        final OlapBuilderGoogleApiDataSource gaDS;

        if (authType == null || authType.getAuthorizationTypeType() == UxBuilderGADataSourceAuthorizationType.AuthorizationType.ServiceAccount)
        {
            gaDS = getDataSourceForServiceAccountUI();
        }
        else
        {
            throw new CdShouldNotBeHereProgrammingException();
        }

        return gaDS;
    }

    @NotNull
    private OlapBuilderGoogleApiDataSource getDataSourceForServiceAccountUI()
    {
        final OlapBuilderGoogleApiDataSource gaDS;
        gaDS = getDataSourceFromUIForServiceAccount();

        final ServiceAccountProperties serviceAccountProperties = getStepModelFor(ServiceAccountProperties.class);
        gaDS.setServiceAccount(serviceAccountProperties);

        final UxBuilderDataSourceNameForm name = getStepModelFor(UxBuilderDataSourceNameForm.class);
        gaDS.setNameAndDescription(name.getName(), name.getDescription());

        return gaDS;
    }

    private UxBuilderWizardValidation onServiceAccountScope(OlapRuntimeContext context, int activeStep)
    {
        final ServiceAccountProperties dataSource = getStepModelFor(ServiceAccountProperties.class);

        try
        {
            final OlapFile file = context.getRootFileSystem().create(dataSource.serviceAccountPrivateKey);
            if (file.isDirectory())
            {
                throw new OlapBuilderErrorException(OlapGoogleApiError.GOOGLE_OAUTH_IO_ERROR, "'" + dataSource.serviceAccountPrivateKey + "' file is a directory");
            }
            if (!file.exists())
            {
                throw new OlapBuilderErrorException(OlapGoogleApiError.GOOGLE_OAUTH_IO_ERROR, "'" + dataSource.serviceAccountPrivateKey + "' file does not exists");
            }
        }
        catch (Exception ex)
        {
            throw new OlapBuilderErrorException(OlapGoogleApiError.GOOGLE_OAUTH_IO_ERROR, "'" + dataSource.serviceAccountPrivateKey + "' is not a valid file");
        }

        final List<UxBuilderWizardStep> newSteps = new ArrayList<>();
        newSteps.add(UxBuilderCreateDataSourceWizardSteps.props(getModel1LocalizationTag(), createScopeModelForService(), false));
        return new UxBuilderWizardValidation(activeStep, newSteps);
    }

    public boolean isGoogleOAuthDisabled(OlapRuntimeContext context)
    {
        return CdStringUtils.isNullOrBlank(context.getUserDefinedProperty(GOOGLE_OAUTH2_PROPERTY));
    }

    protected UxBuilderWizardValidation onServiceAccount(OlapRuntimeContext context, int activeStep)
    {
        return new UxBuilderWizardValidation(
                activeStep,
                Collections.singletonList(UxBuilderCreateDataSourceWizardSteps.props(true, new ServiceAccountProperties()))
        );
    }

    @NotNull
    protected String getModel1LocalizationTag()
    {
        return "steps.model1";
    }

    @NotNull
    protected abstract WEBTOKEN_MODEL1 createWebTokenModel();

    @NotNull
    protected Object createScopeModelForService()
    {
        return createWebTokenModel();
    }

    public abstract OlapBuilderGoogleApiDataSource getDataSourceFromUIForWebtoken();

    protected OlapBuilderGoogleApiDataSource getDataSourceFromUIForServiceAccount()
    {
        return getDataSourceFromUIForWebtoken();
    }

    protected static List<UxBuilderWizardStep> createUxCreateWizardSteps()
    {
        final List<UxBuilderWizardStep> steps = new ArrayList<>();

        steps.add(UxBuilderCreateDataSourceWizardSteps.nameWithValidation());
        steps.add(UxBuilderCreateDataSourceWizardSteps.more() /* depends on previous step */);

        return steps;
    }

    public static class ServiceAccountProperties implements UxAutomodel
    {
        public static final CdProperty SERVICE_ACCOUNT_PRIVATE_KEY = new CdReadWriteProperty(ServiceAccountProperties.class, "serviceAccountPrivateKey", true)
        {
            @Override
            public Class<?> getTypeForDefaultEditor()
            {
                return File.class;
            }
        };

        // file name with the private key for the google service account
        @XmlAttribute()
        protected String serviceAccountPrivateKey;

        public ServiceAccountProperties()
        {

        }
    }
}
