/*
 * Copyright 2014 - 2020 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.googleapi.common;

import com.google.api.client.http.HttpRequestInitializer;
import com.google.api.client.http.HttpTransport;
import com.google.api.client.http.javanet.NetHttpTransport;
import com.google.api.client.json.JsonFactory;
import com.google.api.client.json.gson.GsonFactory;
import com.google.auth.http.HttpCredentialsAdapter;
import com.google.auth.oauth2.GoogleCredentials;
import crazydev.iccube.builder.datasource.OlapBuilderAbstractConnection;
import crazydev.iccube.builder.errors.OlapBuilderErrorCode;
import crazydev.iccube.builder.errors.OlapBuilderErrorException;
import crazydev.iccube.builder.googleapi.authorization.OlapBuilderGoogleCredentialBuilder;
import crazydev.iccube.olap.component.context.OlapRuntimeContext;
import crazydev.iccube.olap.loggers.OlapLoggers;
import crazydev.iccube.request.request.IcCubeRequest;
import org.apache.log4j.Logger;

public abstract class OlapBuilderGoogleApiConnection<T extends Object, DATASOURCE extends OlapBuilderGoogleApiDataSource> extends OlapBuilderAbstractConnection<DATASOURCE>
{
    public static final Logger LOGGER = OlapLoggers.BUILDER_GOOGLE;

    public static final String APPLICATION_NAME = "icCube Local App";

    public static final HttpTransport HTTP_TRANSPORT = new NetHttpTransport();

    /**
     * Global instance of the JSON factory.
     */
    public static final JsonFactory JSON_FACTORY = GsonFactory.getDefaultInstance();

    protected T jsonClient;

    public OlapBuilderGoogleApiConnection(DATASOURCE dataSource)
    {
        super(dataSource);
    }

    public T getJsonClient()
    {
        return jsonClient;
    }

    @Override
    public void onOpen(OlapRuntimeContext context) throws OlapBuilderErrorException
    {
        try
        {
            IcCubeRequest.assertNotCancelled("add-member") /* speed-up cancel processing */;

            // Authorization.
            HttpRequestInitializer initializer = setTimeout(authorize(context));

            IcCubeRequest.assertNotCancelled("add-member") /* speed-up cancel processing */;

            // Set up and return Google Analytics API client.
            jsonClient = buildClient(initializer, HTTP_TRANSPORT, JSON_FACTORY);
        }
        catch (Exception ex)
        {
            throw new OlapBuilderErrorException(ex, OlapBuilderErrorCode.IO_ERROR, ex.getMessage());
        }
    }

    protected HttpRequestInitializer setTimeout(final GoogleCredentials credential)
    {
        final Integer timeout = dataSource.getTimeout();
        if (timeout == null || timeout == 0)
        { // don't allow for infinite timeout
            return new HttpCredentialsAdapter(credential);
        }
        else
        {
            return httpRequest -> {
                new HttpCredentialsAdapter(credential).initialize(httpRequest);
                httpRequest.setConnectTimeout(timeout * 1000);
                httpRequest.setReadTimeout(timeout * 1000);
            };
        }
    }

    protected abstract T buildClient(HttpRequestInitializer initializer, HttpTransport httpTransport, JsonFactory jsonFactory);

    @Override
    protected void onClose()
    {
        jsonClient = null;
    }

    /**
     * Authorizes the installed application to access user's protected data.
     */
    protected GoogleCredentials authorize(OlapRuntimeContext context) throws Exception
    {
        final OlapBuilderGoogleCredentialBuilder builder = dataSource.getCredentialBuilder(context);
        return builder.authorize(context, getAccountScope());
    }

    public abstract String getAccountScope();

}
