/*
 * Copyright 2014 - 2020 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.googleapi.bigquery.datasource;

import com.google.cloud.bigquery.BigQuery;
import com.google.cloud.bigquery.Schema;
import com.google.cloud.bigquery.Table;
import com.google.cloud.bigquery.TableId;
import crazydev.common.property.CdProperty;
import crazydev.common.property.CdReadWriteProperty;
import crazydev.common.utils.CdSql;
import crazydev.common.utils.CdStringUtils;
import crazydev.iccube.builder.OlapBuilderConnectionPool;
import crazydev.iccube.builder.OlapBuilderContext;
import crazydev.iccube.builder.datasource.reader.IOlapBuilderTableRowReader;
import crazydev.iccube.builder.errors.OlapBuilderErrorManager;
import crazydev.iccube.builder.factory.schema.IOlapBuilderJaxbListener;
import crazydev.iccube.builder.model.def.IOlapBuilderDataColumnDef;
import crazydev.iccube.olap.component.context.OlapRuntimeContext;
import jakarta.xml.bind.annotation.XmlAttribute;
import jakarta.xml.bind.annotation.XmlRootElement;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;

@XmlRootElement(name = "goBigqueryTableDT")
public class OlapBuilderGoogleBigqueryDataTable extends OlapBuilderGoogleBigqueryBaseDataTable implements IOlapBuilderJaxbListener
{
    public static final CdProperty TABLEID = new CdReadWriteProperty(OlapBuilderGoogleBigqueryDataTable.class, "tableId", true);

    public static final CdProperty DATASET = new CdReadWriteProperty(OlapBuilderGoogleBigqueryDataTable.class, "datasetId", false);

    public static final CdProperty VERSION = new CdReadWriteProperty(OlapBuilderGoogleBigqueryDataTable.class, "v", false, false);

    public static final CdProperty RESTRICTIONS = new CdReadWriteProperty(OlapBuilderGoogleBigqueryDataTable.class, "restrictions", false, true)
    {
        @Override
        public Class<?> getTypeForDefaultEditor()
        {
            return CdSql.class;
        }
    };

    public static final CdProperty USE_STORAGE_API = new CdReadWriteProperty(OlapBuilderGoogleBigqueryDataTable.class, "useStorageApi", false);

    public static final CdProperty USE_LEGACY_CODE = new CdReadWriteProperty(OlapBuilderGoogleBigqueryDataTable.class, "useLegacyCode", true);

    @XmlAttribute(required = true)
    protected String tableId;

    @XmlAttribute()
    protected String datasetId;

    @XmlAttribute()
    protected Integer v = 1;

    @XmlAttribute()
    protected String restrictions;

    @XmlAttribute()
    private Boolean useLegacyCode;

    @XmlAttribute()
    private Boolean useStorageApi;

    public OlapBuilderGoogleBigqueryDataTable()
    {
    }

    public OlapBuilderGoogleBigqueryDataTable(String tableName, String tableId, String datasetId, boolean useLegacyCode)
    {
        super(tableName);
        this.tableId = tableId;
        this.datasetId = datasetId;
        this.useLegacyCode = useLegacyCode;
    }

    public TableId getApiTableId(OlapBuilderGoogleBigqueryConnection connection)
    {
        final OlapBuilderGoogleBigqueryDataSource ds = connection.getDataSource();

        final String projectId = ds.getProjectId();
        final String dataSetId = getDataSetId(connection);

        return TableId.of(projectId, dataSetId, tableId);
    }

    @Override
    protected String getDataSetId(OlapBuilderGoogleBigqueryConnection connection)
    {
        return CdStringUtils.isNullOrBlank(datasetId) ? super.getDataSetId(connection) : datasetId;
    }

    @Override
    protected IOlapBuilderTableRowReader<OlapBuilderGoogleBigqueryConnection> doCreateFullTableRowReader(OlapBuilderContext context, OlapBuilderConnectionPool connectionPool, int maxRowCount)
    {
        if (useStorageApi())
        {
            return new OlapBuilderGoogleBigQueryStorageDataTableRowReader(context, connectionPool, maxRowCount, this);
        }

        return super.doCreateFullTableRowReader(context, connectionPool, maxRowCount);
    }

    @Override
    protected List<? extends IOlapBuilderDataColumnDef> doDiscoverAllColumns(OlapRuntimeContext context, OlapBuilderGoogleBigqueryConnection openedConnection, OlapBuilderErrorManager errorManager)
    {
        final BigQuery jsonClient = openedConnection.getJsonClient();
        final Table table = jsonClient.getTable(getApiTableId(openedConnection));

        final Schema schema = table.getDefinition().getSchema();

        if (schema == null)
        {
            return super.doDiscoverAllColumns(context, openedConnection, errorManager);
        }
        return toColumns(schema.getFields());
    }

    @Override
    public String getInternalFriendlyTypeName()
    {
        return "Google BigQuery Table";
    }

    @Override
    protected boolean useLegacyCode()
    {
        return useLegacyCode == Boolean.TRUE;
    }

    public boolean useStorageApi()
    {
        return useStorageApi == Boolean.TRUE;
    }

    public void setUseStorageApi(boolean useStorageApi)
    {
        this.useStorageApi = useStorageApi;
    }

    private String addBrackets(String name)
    {
        return useLegacyCode() ? "[" + name + "]" : "`" + name + "`";
    }

    @Override
    protected String buildSelect(OlapBuilderGoogleBigqueryConnection connection, @Nullable Comparable incrLoadMarker)
    {
        final StringBuilder builder = new StringBuilder("SELECT ");

        final List<IOlapBuilderDataColumnDef> selectedColumns = getSelectedColumns();
        for (int i = 0; i < selectedColumns.size(); i++)
        {
            IOlapBuilderDataColumnDef columnDef = selectedColumns.get(i);
            builder.append(i == 0 ? "" : ",").append(addBrackets(columnDef.getName()));
        }
        if (selectedColumns.isEmpty())
        {
            builder.append("*");
        }

        final TableId apiTableId = getApiTableId(connection);
        final String rscName = apiTableId.getProject() + "." + apiTableId.getDataset() + "." + apiTableId.getTable();

        builder.append(" FROM ").append(addBrackets(rscName));

        final boolean hasRestrictions = CdStringUtils.isNotNullAndNotBlank(restrictions);

        if (hasRestrictions)
        {
            if (incrLoadMarker == null)
            {
                builder.append(" WHERE ").append(restrictions);
            }
            else
            {
                builder.append(" WHERE (").append(restrictions).append(")");
            }
        }

        if (incrLoadMarker != null)
        {
            final String restriction = setupIncrLoadRestriction(rscName, incrLoadMarker);

            if (!hasRestrictions)
            {
                builder.append(" WHERE ").append(restriction);
            }
            else
            {
                builder.append(" AND ").append(restriction);
            }
        }

        return builder.toString();
    }

    @Nullable
    public List<String> getRestrictionList()
    {
        if (CdStringUtils.isNullOrBlank(restrictions))
        {
            return null;
        }

        final String[] lines = restrictions.split("\\R");

        if (lines == null || lines.length == 0)
        {
            return null;
        }

        final List<String> statements = new ArrayList<>();

        for (String line : lines)
        {
            if (CdStringUtils.isNotNullAndNotBlank(line))
            {
                statements.add(line.trim());
            }
        }
        return statements;
    }

    @Override
    public void beforeMarshal()
    {

    }

    @Override
    public void afterUnmarshal()
    {
        if (v == null)
        {
            datasetId = TableFormatter.getDataSetId(tableId);
            tableId = TableFormatter.getTableId(tableId);
            v = 1;
        }

    }

    private static class TableFormatter
    {
//        @Nullable
//        public static String getProjectId(String tableId)
//        {
//            final String[] s1 = tableId.split(":");
//            return s1.length < 2 ? null : s1[0] + ":" + s1[1];
//        }

        @Nullable
        public static String getDataSetId(String tableId)
        {
            final String[] s1 = tableId.split(":");
            final String table = s1.length == 0 ? "" : s1[s1.length - 1];
            final String[] s2 = table.split("\\.");
            return s2.length == 1 ? null : s2[0];
        }

        public static String getTableId(String tableId)
        {
            final String[] s2 = tableId.split("\\.");
            return s2.length <= 1 ? tableId : s2[s2.length - 1];
        }
    }
}
