/*
 * Copyright 2014 - 2020 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.googleapi.bigquery.datasource;

import com.google.cloud.bigquery.BigQuery;
import com.google.cloud.bigquery.Dataset;
import com.google.cloud.bigquery.TableId;
import com.google.cloud.bigquery.ViewDefinition;
import crazydev.common.property.CdProperty;
import crazydev.common.property.CdReadWriteProperty;
import crazydev.common.utils.CdStringUtils;
import crazydev.iccube.builder.googleapi.common.OlapBuilderGoogleApiDataSource;
import crazydev.iccube.builder.model.def.IOlapBuilderTabularDataDef;
import crazydev.iccube.builder.model.impl.table.OlapBuilderBaseDataTable;
import crazydev.iccube.builder.ux.meta.datasource.UxBuilderDataSourceType;
import crazydev.iccube.olap.component.context.OlapRuntimeContext;
import jakarta.xml.bind.annotation.XmlAttribute;
import jakarta.xml.bind.annotation.XmlRootElement;
import jakarta.xml.bind.annotation.XmlTransient;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

@XmlRootElement(name = "goBigqueryDS")
public class OlapBuilderGoogleBigqueryDataSource extends OlapBuilderGoogleApiDataSource<OlapBuilderGoogleBigqueryConnection>
{
    public static final CdProperty PROJECT = new CdReadWriteProperty(OlapBuilderGoogleBigqueryDataSource.class, "projectId", true);

    public static final CdProperty CREDENTIALS_PROJECT = new CdReadWriteProperty(OlapBuilderGoogleBigqueryDataSource.class, "credentialsProjectId", false)
    {
        @Override
        public String getGroupName()
        {
            return CREDENTIALS;
        }
    };

    public static final CdProperty DATASET = new CdReadWriteProperty(OlapBuilderGoogleBigqueryDataSource.class, "datasetId", false);

    public static final CdProperty TIME_OUT = new CdReadWriteProperty(OlapBuilderGoogleBigqueryDataSource.class, "waitTime", false);

    public static final CdProperty TOKEN_SUBSCOPE = new CdReadWriteProperty(OlapBuilderGoogleBigqueryDataSource.class, "tokenSubscope", false);

    final static List<Class<? extends IOlapBuilderTabularDataDef>> CREATED_TABLE_TYPES = Arrays.asList(
            OlapBuilderGoogleBigqueryStatementDataTable.class
    );

    @XmlTransient
    private final Map<String, TableInfo> discoveredTables = new ConcurrentHashMap<>();

    @XmlAttribute(required = false)
    public String projectId;

    @XmlAttribute(required = false)
    public String credentialsProjectId;

    @XmlAttribute(required = false)
    public String datasetId;

    @XmlTransient
    private long discoveredTablesMs = -1;

    @Nullable
    @XmlAttribute(required = false)
    private Integer waitTime;

    @XmlAttribute(required = false)
    private String tokenSubscope;

    public OlapBuilderGoogleBigqueryDataSource()
    {
    }

    public OlapBuilderGoogleBigqueryDataSource(String serviceAccountPrivateKey)
    {
        super(serviceAccountPrivateKey);
    }

    @Override
    protected String getReportDataSourceType()
    {
        return "googleBigQuery";
    }

    @Override
    protected String getReportDataSourceTypeCaption()
    {
        return "Google Big Query";
    }

    public String getProjectId()
    {
        return projectId;
    }

    public String getCredentialsProjectId()
    {
        return CdStringUtils.isNotNullAndNotBlank(credentialsProjectId) ? credentialsProjectId : projectId;
    }

    public void setProjectId(String projectId)
    {
        this.projectId = projectId;
    }

    public String getDatasetId()
    {
        return datasetId;
    }

    public void setDatasetId(String datasetId)
    {
        this.datasetId = datasetId;
    }

    @Override
    public boolean isDiscoverTablesSupported()
    {
        return true;
    }

    /**
     * @return a possibly null list of table that can be created by this datasource.
     */
    @Override
    @Nullable
    public List<Class<? extends IOlapBuilderTabularDataDef>> getCreatedTableTypes()
    {
        return CREATED_TABLE_TYPES;
    }

    @Override
    public OlapBuilderGoogleBigqueryConnection createConnection(OlapRuntimeContext context, boolean forceRefresh)
    {
        return new OlapBuilderGoogleBigqueryConnection(this);
    }

    @Override
    public OlapBuilderGoogleBigqueryBaseDataTable createEmptyDiscoverTable(String tableName)
    {
        final TableInfo info = discoveredTables.get(tableName);
        if (info == null)
        {
            throw new RuntimeException("Unable to find table '" + tableName + "' in definition");
        }

        return new OlapBuilderGoogleBigqueryDataTable(tableName,
                                                      info.tableName,
                                                      nullIfSame(info.dsName, datasetId), info.useLegacy);
    }

    private static String nullIfSame(String first, String check)
    {
        return first == null || first.equals(check) ? null : first;
    }

    @Override
    public OlapBuilderBaseDataTable<OlapBuilderGoogleBigqueryConnection> createEmptyCreateTable(@Nullable Class<IOlapBuilderTabularDataDef> tableType)
    {
        return new OlapBuilderGoogleBigqueryStatementDataTable();
    }

    @Override
    public List<String> discoverAllTablesNames(OlapBuilderGoogleBigqueryConnection connection, boolean filterSystemSchemas, final @Nullable String filter)
    {
        final List<String> tableNames = new ArrayList<>();

        synchronized (this)
        {
            if (discoveredTablesMs > 0 && System.currentTimeMillis() - discoveredTablesMs < 60 * 1000)
            {
                discoveredTables.keySet().forEach(tableName -> {
                    if (matches(filter, tableName))
                    {
                        tableNames.add(tableName);
                    }
                });

                discoveredTablesMs = System.currentTimeMillis();
            }
            else
            {
                discoveredTables.clear();

                final BigQuery bigQuery = connection.createJsonClientForDiscover();

                if (CdStringUtils.isNotNullAndNotBlank(datasetId))
                {
                    final Dataset dataset = bigQuery.getDataset(datasetId);
                    discoverTables(dataset, tableNames, filter, true);
                }
                else
                {
                    bigQuery.listDatasets().iterateAll().forEach(dataset -> {
                        discoverTables(dataset, tableNames, filter, false);
                    });
                }
                discoveredTablesMs = -1;
                if (CdStringUtils.isNullOrBlank(filter))
                {
                    discoveredTablesMs = System.currentTimeMillis();
                }
            }
        }

        return tableNames;
    }

    private void discoverTables(Dataset dataset, List<String> tableNames, @Nullable String filter, boolean singleDataSet)
    {
        dataset.list().iterateAll().forEach(table -> {

            boolean useLegacy = false;
            if (table.getDefinition() instanceof ViewDefinition)
            {
                ViewDefinition definition = table.getDefinition();
                useLegacy = definition.useLegacySql() == Boolean.TRUE;
            }

            final TableId tableId = table.getTableId();
            final String tableNameForUser = (singleDataSet ? "" : tableId.getDataset() + ".") + tableId.getTable();
            if (matches(filter, tableNameForUser))
            {
                discoveredTables.put(tableNameForUser, new TableInfo(tableId, useLegacy));
                tableNames.add(tableNameForUser);
            }
        });
    }

    private boolean matches(@Nullable String filter, String tableNameForUser)
    {
        return filter == null || tableNameForUser.contains(filter);
    }

    @Nullable
    public Integer getWaitTime()
    {
        return waitTime;
    }

    @Override
    protected UxBuilderDataSourceType createUxType()
    {
        return new UxBuilderDataSourceType(getTypeId(), getUxTypeImage(), () -> new OlapBuilderGoogleBigQueryDataSourceUxWizard(this, getTypeId()));
    }

    public String getTokenSubscope()
    {
        return tokenSubscope;
    }

    public void setTokenSubscope(String bqTokenSubscope)
    {
        this.tokenSubscope = bqTokenSubscope;
    }

    private static class TableInfo
    {
        final String tableName;

        final String dsName;

        final boolean useLegacy;

        public TableInfo(TableId tableId, boolean useLegacy)
        {
            tableName = tableId.getTable();
            dsName = tableId.getDataset();
            this.useLegacy = useLegacy;
        }
    }
}
