/*
 * Copyright 2014 - 2020 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.googleapi.bigquery.datasource;

import com.google.cloud.bigquery.storage.v1.ArrowRecordBatch;
import com.google.cloud.bigquery.storage.v1.ArrowSchema;
import crazydev.iccube.builder.model.def.IOlapBuilderDataColumnDef;
import crazydev.iccube.builder.model.def.IOlapBuilderTableRow;
import crazydev.iccube.builder.model.def.OlapBuilderAbstractTableRow;
import org.apache.arrow.memory.BufferAllocator;
import org.apache.arrow.memory.RootAllocator;
import org.apache.arrow.vector.FieldVector;
import org.apache.arrow.vector.VectorLoader;
import org.apache.arrow.vector.VectorSchemaRoot;
import org.apache.arrow.vector.ipc.ReadChannel;
import org.apache.arrow.vector.ipc.message.MessageSerializer;
import org.apache.arrow.vector.types.pojo.ArrowType;
import org.apache.arrow.vector.types.pojo.Field;
import org.apache.arrow.vector.types.pojo.Schema;
import org.apache.arrow.vector.util.ByteArrayReadableSeekableByteChannel;
import org.apache.arrow.vector.util.Text;
import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;

import java.io.IOException;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

public class OlapBuilderGoogleBigQueryStorageArrowResponseDecoder
{
    private final BufferAllocator allocator = new RootAllocator(Long.MAX_VALUE);

    private final VectorSchemaRoot root;

    private final VectorLoader loader;

    private final int batchRowCount;

    private int nextRow;

    public OlapBuilderGoogleBigQueryStorageArrowResponseDecoder(ArrowSchema arrowSchema, ArrowRecordBatch batch)
            throws IOException
    {
        final Schema schema = MessageSerializer.deserializeSchema(
                new ReadChannel(new ByteArrayReadableSeekableByteChannel(arrowSchema.getSerializedSchema().toByteArray()))
        );

        final List<FieldVector> vectors = new ArrayList<>();
        final List<Field> fields = schema.getFields();

        for (Field field : fields)
        {
            vectors.add(field.createVector(allocator));
        }

        root = new VectorSchemaRoot(vectors);
        loader = new VectorLoader(root);

        org.apache.arrow.vector.ipc.message.ArrowRecordBatch deserializedBatch = MessageSerializer.deserializeRecordBatch(
                new ReadChannel(new ByteArrayReadableSeekableByteChannel(batch.getSerializedRecordBatch().toByteArray())), allocator
        );

        loader.load(deserializedBatch);

        deserializedBatch.close();

        batchRowCount = root.getRowCount();
        nextRow = 0;
    }

    @Nullable
    public IOlapBuilderTableRow nextRow()
    {
        if (nextRow /* the row to return */ >= batchRowCount)
        {
            return null;
        }

        final int row = nextRow++;

        final List<FieldVector> vectors = root.getFieldVectors();

        return new OlapBuilderAbstractTableRow()
        {
            @Override
            @Nullable
            protected Object getDataImpl(IOlapBuilderDataColumnDef columnDef)
            {
                final int columnIdx = columnDef.getIndex();

                final FieldVector vector = vectors.get(columnIdx);

                final Object value = vector.getObject(row);

                if (value == null)
                {
                    return null;
                }

                final ArrowType.ArrowTypeID typeID = vector.getField().getFieldType().getType().getTypeID();

                switch (typeID)
                {
                    case Utf8:
                    case LargeUtf8:
                    {
                        final Text text = (Text) value;
                        return text.toString();
                    }
                    case Timestamp:
                    {
                        long epoch;
                        if (value instanceof java.time.LocalDateTime)
                        {
                            // epoch = ((java.time.LocalDateTime) value).toEpochSecond(ZoneOffset.UTC) * 1000;
                            epoch = ((java.time.LocalDateTime) value).toInstant(ZoneOffset.UTC).toEpochMilli();
                        }
                        else
                        {
                            epoch = (Long) value / 1000;
                        }
                        return new LocalDateTime(new Date(epoch), DateTimeZone.UTC);
                    }
                    case Date:
                    {
                        // 32-bit days since epoch
                        final int days = (Integer) value;
                        final long epoch = days * 86400L * 1000L;
                        return new LocalDate(new Date(epoch), DateTimeZone.UTC);
                    }
                    case Time:
                    {
                        return ((Number) value).longValue();
                    }
                }

                return value instanceof Comparable ? (Comparable) value : null;
            }
        };
    }

    public void close()
    {
        root.close();
        allocator.close();
    }

}
