/*
 * Copyright 2014 - 2020 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.googleapi.bigquery.datasource;

import com.google.api.client.auth.oauth2.TokenResponseException;
import com.google.api.client.googleapis.json.GoogleJsonResponseException;
import com.google.cloud.bigquery.*;
import crazydev.iccube.builder.OlapBuilderConnectionPool;
import crazydev.iccube.builder.OlapBuilderContext;
import crazydev.iccube.builder.datasource.reader.OlapBuilderAbstractTableRowReader;
import crazydev.iccube.builder.errors.OlapBuilderErrorException;
import crazydev.iccube.builder.googleapi.errors.OlapGoogleApiError;
import crazydev.iccube.builder.model.def.IOlapBuilderDataColumnDef;
import crazydev.iccube.builder.model.def.IOlapBuilderTableRow;
import crazydev.iccube.builder.model.def.OlapBuilderAbstractTableRow;
import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;

import java.io.IOException;
import java.time.LocalTime;
import java.util.Date;
import java.util.Iterator;

public class OlapBuilderGoogleBigQueryDataTableRowReader
        extends OlapBuilderAbstractTableRowReader<OlapBuilderGoogleBigqueryBaseDataTable, OlapBuilderGoogleBigqueryConnection>
{
    private int rowsCount;

    private Iterator<FieldValueList> result;

    @Nullable
    private Schema schema;

    public OlapBuilderGoogleBigQueryDataTableRowReader(OlapBuilderContext context, OlapBuilderConnectionPool connectionPool, int maxRowCount, OlapBuilderGoogleBigqueryBaseDataTable table)
    {
        super(context, connectionPool, maxRowCount, table, table.getName());
    }

    @Override
    public void doInit()
    {
        initConnection();

        rowsCount = 0;

        try
        {
            final Comparable incrLoadMarker = context.getIncrementalLoadMarker(table);

            final TableResult tableResult = table.executeQuery(connection, maxRowCount, table, incrLoadMarker);
            schema = tableResult.getSchema();
            result = tableResult.iterateAll().iterator();
        }
        catch (GoogleJsonResponseException ex)
        {
            throw OlapGoogleApiError.error("Table", getTableName(), ex);
        }
        catch (TokenResponseException ex)
        {
            throw OlapGoogleApiError.error("Table", getTableName(), ex);
        }
        catch (IOException | InterruptedException ex)
        {
            throw new OlapBuilderErrorException(true, OlapGoogleApiError.IO_ERROR, ex.getLocalizedMessage(), "");
        }

    }

    @Override
    public boolean isRowSafe()
    {
        return false;
    }

    @Override
    public IOlapBuilderTableRow doNextRow()
    {
        // reach the limit
        if (maxRowCount != -1 && rowsCount >= maxRowCount)
        {
            return null;
        }

        if (!result.hasNext())
        {
            return null;
        }

        rowsCount++;
        final FieldValueList row = result.next();

        if (schema != null)
        {
            return new OlapBuilderAbstractTableRow()
            {
                @Override
                @Nullable
                protected Object getDataImpl(IOlapBuilderDataColumnDef columnDef)
                {
                    final FieldValue value = row.get(columnDef.getName());
                    if (value.isNull())
                    {
                        return null;
                    }
                    final Field fieldDef = schema.getFields().get(columnDef.getName());
                    switch (fieldDef.getType().getStandardType())
                    {
                        case BOOL:
                            return value.getBooleanValue();
                        case INT64:
                            return value.getLongValue();
                        case NUMERIC:
                        case FLOAT64:
                            return value.getDoubleValue();
                        case GEOGRAPHY:
                        case STRING:
                            return value.getStringValue();
                        case TIMESTAMP:
                            return toJodaTimeStamp(value);
                        case DATETIME:
                            return new LocalDateTime(value.getValue());
                        case DATE:
                            return new LocalDate(value.getValue());
                        case TIME:
                            final LocalTime time = LocalTime.parse(value.getStringValue());
                            return time.toNanoOfDay() / 1000;
                        default:
                            return toComparable(value);
                    }
                }
            };
        }
        else
        {
            return new OlapBuilderAbstractTableRow()
            {
                @Override
                @Nullable
                protected Object getDataImpl(IOlapBuilderDataColumnDef columnDef)
                {
                    final FieldValue value = row.get(columnDef.getName());
                    if (value.isNull())
                    {
                        return null;
                    }
                    switch (columnDef.getType())
                    {
                        case DATE:
                            return toJodaDate(value);
                        case DATETIME:
                            return toJodaDateTime(value);
                        default:
                            return toComparable(value);
                    }
                }
            };
        }
    }

    public Comparable toComparable(FieldValue fieldValue)
    {
        final Object value = fieldValue.getValue();
        return value instanceof Comparable ? (Comparable) value : null;
    }

    private LocalDateTime toJodaTimeStamp(FieldValue value)
    {
        final long epoch = value.getTimestampValue() / 1000;
        return new LocalDateTime(new Date(epoch), DateTimeZone.UTC);
    }

    private LocalDate toJodaDate(FieldValue value)
    {
        try
        {
            final long epoch = value.getTimestampValue() / 1000;
            return new LocalDate(new Date(epoch), DateTimeZone.UTC);
        }
        catch (NumberFormatException ex)
        {
            return new LocalDate(value.getValue());
        }
    }

    private LocalDateTime toJodaDateTime(FieldValue value)
    {
        try
        {
            return toJodaTimeStamp(value);
        }
        catch (NumberFormatException ex)
        {
            return new LocalDateTime(value.getValue());
        }
    }

    /**
     * Will be called as soon as the init method has been called (either successfully or on error).
     * Keep it safe as it is typically called from a finally block.
     */
    @Override
    public void done()
    {
        super.done();
    }


}
