/*
 * Copyright 1999 - 2014 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.mongodb.datatable.mapreduce;

import com.mongodb.*;
import crazydev.iccube.builder.OlapBuilderContext;
import crazydev.iccube.builder.mongodb.datasource.OlapBuilderMongoDbConnection;
import org.jetbrains.annotations.Nullable;

import java.util.Iterator;

public abstract class OlapBuilderMongoDbMapReduceHelper
{
    public static Cursor mapReduce(OlapBuilderContext context,
                                   OlapBuilderMongoDbConnection connection,
                                   OlapBuilderMongoDbMapReduceDataTable table)
    {
        return mapReduce(context, connection, table, null);
    }

    @Nullable
    public static DBObject mapReduceOne(OlapBuilderMongoDbConnection connection,
                                        OlapBuilderMongoDbMapReduceDataTable table)
    {
        final Cursor cursor = mapReduce(null, connection, table, 1);

        try
        {
            if (cursor.hasNext())
            {
                final DBObject one = cursor.next();
                return one;
            }

            return null;
        }
        finally
        {
            cursor.close();
        }
    }

    /**
     * Execute the whole list of map/reduce commands and return the result of the last one.
     */
    public static Cursor mapReduce(@Nullable OlapBuilderContext context,
                                   OlapBuilderMongoDbConnection connection,
                                   OlapBuilderMongoDbMapReduceDataTable table,
                                   @Nullable Integer forceBatchSize)
    {
        // The MongoDB shell syntax is as following :
        //
        //     var mapF    = function() {emit({type:'all'},{count:1})};
        //     var reduceF = function(key, values) {return {count:values.length}},
        //
        //     db.runCommand({
        //
        //         mapReduce : 'org',
        //         map       : mapF,
        //         reduce    : reduceF,
        //         out       : {inline:1}
        //
        //     })
        //
        // It must be a bit adapted to be used a JAVA command :
        //
        //     - functions must be placed inline
        //     - functions must be wrapped into string (and properly escaped)
        //
        //     {
        //         mapReduce : 'org',
        //
        //         map       : " function() {emit({type:'all'},{count:1})} ",
        //                     ^                        ^
        //                     as string                ^
        //                                              single-quote (or escape...)
        //
        //         reduce    : " function(key, values) {return {count:values.length}} "
        //
        //         out       : {inline:1}
        //     }
        //

//        final Comparable incrLoadMarker = context != null ? context.getIncrementalLoadMarker(table) : null;
//
//        final List<DBObject> commands;
//
//        if (incrLoadMarker == null)
//        {
//            commands = table.getMapReduceCommandsAsDBObject();
//        }
//        else
//        {
//            commands = table.getMapReduceCommandsIncrLoadAsDBObject(incrLoadMarker);
//        }
//
//        if (commands.isEmpty())
//        {
//            final MongoClient mongo = connection.getMongoClient();
//            return new EmptyCursor(mongo.getAddress());
//        }
//
//        MapReduceOutput lastOutput = null;
//
//        for (int ii = 0; ii < commands.size(); ii++)
//        {
//            final DB mongoDB = connection.getMongoDB();
//
//            final DBObject command = commands.get(ii);
//
//            final MapReduceOutput mapReduceOutput = OlapBuilderMongoDbMapReduceData.mapReduce(mongoDB, command);
//
//            if (ii == commands.size() - 1 /* last map/reduce command */)
//            {
//                lastOutput = mapReduceOutput;
//            }
//        }
//
//        final Cursor cursor = asCursor(connection.getMongoClient(), table, forceBatchSize, lastOutput);
        return null;
    }

    private static Cursor asCursor(MongoClient mongo, OlapBuilderMongoDbMapReduceDataTable table, @Nullable Integer forceBatchSize, @Nullable MapReduceOutput lastOutput)
    {
        if (lastOutput == null)
        {
            return new EmptyCursor(mongo.getAddress());
        }

        final Iterable<DBObject> res_ = lastOutput.results();

        if (res_ == null)
        {
            return new EmptyCursor(mongo.getAddress());
        }

        if (res_ instanceof DBCursor)
        {
            final DBCursor cursor = (DBCursor) res_;

            final Integer skip = table.getSkip();

            if (skip != null)
            {
                cursor.skip(skip);
            }

            final Integer limit = table.getLimit();

            if (limit != null)
            {
                cursor.limit(limit);
            }

            final Integer batchSize = forceBatchSize != null ? forceBatchSize : table.getBatchSize();

            if (batchSize != null)
            {
                cursor.batchSize(batchSize);
            }

            return cursor;
        }

        // Results are INLINE

        return new InlineCursor(mongo.getAddress(), res_);
    }

    static class InlineCursor implements Cursor
    {
        private final ServerAddress serverAddress;

        private final Iterator<DBObject> items;

        public InlineCursor(ServerAddress serverAddress, Iterable<DBObject> items)
        {
            this.serverAddress = serverAddress;
            this.items = items.iterator();
        }

        @Override
        public long getCursorId()
        {
            return 0 /* not used by icCube readers ... */;
        }

        @Override
        public ServerAddress getServerAddress()
        {
            return serverAddress;
        }

        @Override
        public boolean hasNext()
        {
            return items.hasNext();
        }

        @Override
        public DBObject next()
        {
            return items.next();
        }

        @Override
        public void remove()
        {
            items.remove();
        }

        @Override
        public void close()
        {
        }

    }

    static class EmptyCursor implements Cursor
    {
        private final ServerAddress serverAddress;

        public EmptyCursor(ServerAddress serverAddress)
        {
            this.serverAddress = serverAddress;
        }

        @Override
        public long getCursorId()
        {
            return 0 /* not used by icCube readers ... */;
        }

        @Override
        public ServerAddress getServerAddress()
        {
            return serverAddress;
        }

        @Override
        public boolean hasNext()
        {
            return false;
        }

        @Override
        public DBObject next()
        {
            return null;
        }

        @Override
        public void remove()
        {
        }

        @Override
        public void close()
        {
        }

    }

}
