/*
 * Copyright 1999 - 2015 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.mongodb.datatable.mapreduce;

import com.mongodb.*;
import crazydev.iccube.builder.errors.OlapBuilderErrorException;
import crazydev.iccube.builder.mongodb.error.OlapBuilderMongoDbErrorCode;
import crazydev.iccube.olap.loggers.OlapLoggers;
import org.jetbrains.annotations.Nullable;

import java.util.Map;
import java.util.concurrent.TimeUnit;

public class OlapBuilderMongoDbMapReduceData
{
    private final String inputCollectionName;

    private final String map;

    private final String reduce;

    private final String outputCollection;

    private final MapReduceCommand.OutputType type;

    @Nullable
    private final DBObject query;

    public OlapBuilderMongoDbMapReduceData(String inputCollectionName,
                                           String map,
                                           String reduce,
                                           String outputCollection,
                                           MapReduceCommand.OutputType type,
                                           @Nullable DBObject query
    )
    {
        this.inputCollectionName = inputCollectionName;

        this.map = map;
        this.reduce = reduce;

        this.outputCollection = outputCollection;
        this.type = type;

        this.query = query;
    }

    public static MapReduceOutput mapReduce(DB mongoDB, DBObject object)
    {
        if (OlapLoggers.BUILDER_MONGODB.isDebugEnabled())
        {
            OlapLoggers.BUILDER_MONGODB.debug("[MongoDB] mapReduce (json): " + object.toString());
        }

        final OlapBuilderMongoDbMapReduceData data = create(object);

        final DBCollection inputCollection = mongoDB.getCollection(data.inputCollectionName);

        if (inputCollection == null)
        {
            throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.ERROR, "the mapReduce collection [" + data.inputCollectionName + "] does not exist");
        }

        final MapReduceCommand command = new MapReduceCommand(
                inputCollection,
                data.map,
                data.reduce,
                data.outputCollection,
                data.type,
                data.query
        );

        final String finalize = asOptionalString(object, "finalize");

        final DBObject sort = asOptionalObject(object, "sort");
        final int limit = asOptionalInteger(object, "limit");

        final Map<String, Object> scope = asOptionalMap(object, "scope");
        final Boolean jsMode = asOptionalBoolean(object, "jsMode");
        final Boolean verbose = asOptionalBoolean(object, "verbose");
        final Boolean bypassDocumentValidation = asOptionalBoolean(object, "bypassDocumentValidation");

        final String outputDB = asOptionalString(object, "db");
        final long maxTimeMS = asOptionalInteger(object, "maxTimeMS");

        command.setFinalize(finalize);
        command.setSort(sort);
        command.setLimit(limit);

        command.setScope(scope);
        command.setJsMode(jsMode);

        if (verbose != null)
        {
            command.setVerbose(verbose);
        }

        command.setBypassDocumentValidation(bypassDocumentValidation);

        command.setOutputDB(outputDB);
        command.setMaxTime(maxTimeMS, TimeUnit.MILLISECONDS);

        if (OlapLoggers.BUILDER_MONGODB.isDebugEnabled())
        {
            OlapLoggers.BUILDER_MONGODB.debug("[MongoDB] mapReduce ( cmd): " + command.toString());
        }

        final MapReduceOutput res = inputCollection.mapReduce(command);
        return res;
    }

    public static OlapBuilderMongoDbMapReduceData create(DBObject object)
    {
        final String inputCollectionName = asString(object, "mapReduce");

        final String map = asString(object, "map");
        final String reduce = asString(object, "reduce");

        final String outputCollection = asOutputCollection(object);
        final MapReduceCommand.OutputType outputType = asOutputType(object);

        final DBObject query = asOptionalObject(object, "query");

        return new OlapBuilderMongoDbMapReduceData(
                inputCollectionName,
                map,
                reduce,
                outputCollection,
                outputType,
                query
        );
    }

    private static String asString(DBObject object, String key)
    {
        final Object value = object.get(key);

        if (value instanceof String)
        {
            return (String) value;
        }

        if (value == null)
        {
            throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.ERROR, "the [" + key + "] is not specified");
        }

        throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.ERROR, "the [" + key + "] must be a string");
    }

    @Nullable
    private static String asOptionalString(DBObject object, String key)
    {
        final Object value = object.get(key);

        if (value instanceof String)
        {
            return (String) value;
        }

        if (value == null)
        {
            return null;
        }

        throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.ERROR, "the [" + key + "] must be a string");
    }

    @Nullable
    private static Boolean asOptionalBoolean(DBObject object, String key)
    {
        final Object value = object.get(key);

        if (value instanceof Boolean)
        {
            return (Boolean) value;
        }

        if (value == null)
        {
            return null;
        }

        throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.ERROR, "the [" + key + "] must be a boolean");
    }

    private static int asOptionalInteger(DBObject object, String key)
    {
        final Object value = object.get(key);

        if (value instanceof Long || value instanceof Integer || value instanceof Byte)
        {
            return (int) value;
        }

        if (value == null)
        {
            return 0;
        }

        throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.ERROR, "the [" + key + "] must be a integer");
    }

    private static MapReduceCommand.OutputType asOutputType(DBObject object)
    {
        final Object out = object.get("out");

        if (out == null)
        {
            throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.ERROR, "the out option is not specified");
        }

        if (out instanceof String /* output to a collection */)
        {
            return MapReduceCommand.OutputType.REPLACE;
        }

        if (out instanceof DBObject)
        {
            final DBObject dbObject = (DBObject) out;

            if (dbObject.containsField("inline"))
            {
                return MapReduceCommand.OutputType.INLINE;
            }

            if (dbObject.containsField("replace"))
            {
                return MapReduceCommand.OutputType.REPLACE;
            }

            if (dbObject.containsField("merge"))
            {
                return MapReduceCommand.OutputType.MERGE;
            }

            if (dbObject.containsField("reduce"))
            {
                return MapReduceCommand.OutputType.REDUCE;
            }
        }

        throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.ERROR, "the out option is invalid (type)");
    }

    private static String asOutputCollection(DBObject object)
    {
        final Object out = object.get("out");

        if (out == null)
        {
            throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.ERROR, "the out option is not specified");
        }

        if (out instanceof String /* output to a collection */)
        {
            return (String) out;
        }

        if (out instanceof DBObject)
        {
            final DBObject dbObject = (DBObject) out;

            if (dbObject.containsField("inline"))
            {
                return "";
            }

            if (dbObject.containsField("replace"))
            {
                return asString(dbObject, "replace");
            }

            if (dbObject.containsField("merge"))
            {
                return asString(dbObject, "merge");
            }

            if (dbObject.containsField("reduce"))
            {
                return asString(dbObject, "reduce");
            }
        }

        throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.ERROR, "the out option is invalid (type)");
    }

    @Nullable
    private static DBObject asOptionalObject(DBObject object, String key)
    {
        final Object value = object.get(key);

        if (value instanceof DBObject)
        {
            return (DBObject) value;
        }

        if (value == null)
        {
            return null;
        }

        throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.ERROR, "the [" + key + "] must be a document");
    }

    @Nullable
    private static Map<String, Object> asOptionalMap(DBObject object, String key)
    {
        final Object value = object.get(key);

        if (value instanceof Map)
        {
            return (Map<String, Object>) value;
        }

        if (value == null)
        {
            return null;
        }

        throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.ERROR, "the [" + key + "] must be a document");
    }

    public String getMap()
    {
        return map;
    }

    public String getReduce()
    {
        return reduce;
    }

}
