/*
 * Copyright 2014 - 2019 icCube Software Llc.
 *
 * The code and all underlying concepts and data models are owned fully
 * and exclusively by icCube Software Llc. and are protected by
 * copyright law and international treaties.
 *
 * Warning: Unauthorized reproduction, use or distribution of this
 * program, concepts, documentation and data models, or any portion of
 * it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
 */
package crazydev.iccube.builder.mongodb.datatable.common;

import com.iccube.bson.ic3BsonCodecRegistry;
import com.iccube.bson.ic3BsonRow;
import com.iccube.bson.ic3BsonRowBuilder;
import com.mongodb.BasicDBObject;
import com.mongodb.MongoException;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoCursor;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.Filters;
import crazydev.common.utils.CdStringUtils;
import crazydev.iccube.builder.OlapBuilderConnectionPool;
import crazydev.iccube.builder.OlapBuilderContext;
import crazydev.iccube.builder.datasource.reader.IOlapBuilderTablePartitionKey;
import crazydev.iccube.builder.datasource.reader.OlapBuilderAbstractTablePartitionKey;
import crazydev.iccube.builder.datasource.reader.OlapBuilderAbstractTableRowReader;
import crazydev.iccube.builder.errors.OlapBuilderErrorException;
import crazydev.iccube.builder.model.def.IOlapBuilderDataColumnDef;
import crazydev.iccube.builder.model.def.IOlapBuilderTableRow;
import crazydev.iccube.builder.mongodb.common.OlapBuilderMongoDbLoggers;
import crazydev.iccube.builder.mongodb.datasource.OlapBuilderMongoDbConnection;
import crazydev.iccube.builder.mongodb.error.OlapBuilderMongoDbErrorCode;
import crazydev.iccube.configuration.component.properties.OlapProcessingFactsMode;
import org.bson.BSONException;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.bson.json.JsonParseException;
import org.bson.types.BasicBSONList;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public abstract class OlapBuilderMongoDbTableRowReader<TABLE extends OlapBuilderMongoDbDataTable>
        extends OlapBuilderAbstractTableRowReader<TABLE, OlapBuilderMongoDbConnection>
{
    protected MongoCursor<ic3BsonRow> result;

    protected OlapBuilderAbstractTablePartitionKey partitionKey;

    protected OlapBuilderMongoDbTableRowReader(OlapBuilderContext context, OlapBuilderConnectionPool connectionPool, int maxRowCount, TABLE table, String fullNameForEndUser)
    {
        super(context, connectionPool, maxRowCount, table, fullNameForEndUser);
    }

    @Override
    public boolean isRowSafe()
    {
        return true;
    }

    @Override
    public void init()
    {
        try
        {
            super.init();
        }
        catch (MongoException ex)
        {
            // timeout or similar
            throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.UNEXPECTED_ERROR, ex.getMessage());
        }
    }

    @Override
    @Nullable
    public IOlapBuilderTableRow doNextRow()
    {
        if (!result.hasNext())
        {
            return null;
        }

        final ic3BsonRow next = result.next();
        return next;
    }

    @Override
    public void done()
    {
        if (result != null)
        {
            try
            {
                result.close();
            }
            catch (Exception ignore)
            {
                //yes
            }
        }

        super.done();
    }

    @Override
    @Nullable
    public IOlapBuilderTablePartitionKey getPartitionKey()
    {
        return partitionKey;
    }

    public MongoCollection<Document> getCollection(String collectionName)
    {
        final OlapProcessingFactsMode mode = context.getEngineContext().getOlapEngineProperties().getLoadProcessingFactsMode();

        MongoDatabase db = connection.getMongoClient().getDatabase(connection.getDataSource().getDbName());

        final ic3BsonRowBuilder rowBuilder = buildRows(table.getAllColumns());
        // ic3BsonRowBuilder rowBuilder = buildTree(table.getSelectedColumns());
        db = db.withCodecRegistry(new ic3BsonCodecRegistry(db.getCodecRegistry(), rowBuilder, table.getCacheStringValue(), mode));

        return db.getCollection(collectionName);
    }

    private ic3BsonRowBuilder buildRows(List<IOlapBuilderDataColumnDef> colNames)
    {
        ic3BsonRowBuilder rootNode = new ic3BsonRowBuilder(new Comparable[colNames.size()]);
        colNames.forEach(col -> {
            final String colName = col.getName();
            final String[] names = colName.split("\\.");
            ic3BsonRowBuilder node = rootNode;
            for (String name : names)
            {
                node = node.getOrCreateChild(name);
            }
            node.setIndex(col);
        });
        return rootNode;
    }

    public static BasicDBObject toBson(String fieldName, @Nullable String json)
    {
        try
        {
            return CdStringUtils.isNullOrBlank(json) ? new BasicDBObject() : BasicDBObject.parse(json);
        }
        catch (BSONException | JsonParseException ex)
        {
            throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.JSON_ERROR, "[field:" + fieldName + "] " + ex.getMessage(), json);
        }
    }

    public static List<Bson> toBsonList(String fieldName, String kind, @Nullable String value)
    {
        return toBsonList(fieldName, kind, value, false);

    }

    public static List<Bson> toBsonList(String fieldName, String kind, @Nullable String value, boolean forValidation)
    {
        final String json = CdStringUtils.isNotNullAndNotBlank(value) ? value : null;

        if (json == null)
        {
            return Collections.emptyList();
        }

        value = value.trim();

        if (!value.startsWith("["))
        {
            value = "[" + value;
        }

        if (!value.endsWith("]"))
        {
            value = value + "]";
        }

        try
        {
            final BasicDBObject bson = BasicDBObject.parse("{ a: " + value + "}");
            final Object list = bson.get("a");

            if (list instanceof BasicBSONList)
            {
                final List<Bson> retList = new ArrayList<>();
                final BasicBSONList bList = (BasicBSONList) list;

                for (int i = 0; i < bList.size(); i++)
                {
                    final Object object = bList.get(i);
                    if (object instanceof Bson)
                    {
                        Bson miniBson = (Bson) object;
                        retList.add(miniBson);
                    }
                    else
                    {
                        if (forValidation)
                        {
                            throw new RuntimeException("[field:" + fieldName + "] incorrect type, BSON expected " + object.getClass().getSimpleName());
                        }
                        else
                        {
                            throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.UNEXPECTED_ERROR, "[field:" + fieldName + "] incorrect type, BSON expected " + object.getClass().getSimpleName());
                        }
                    }
                }
                return retList;
            }
            if (forValidation)
            {
                throw new RuntimeException("Internal Error");
            }
            else
            {
                throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.UNEXPECTED_ERROR, "should not be here");
            }
        }
        catch (RuntimeException ex)
        {
            if (forValidation)
            {
                throw ex;
            }
            else
            {
                OlapBuilderMongoDbLoggers.GENERAL.error("[MongoDB] [field:" + fieldName + "] unexpected JSON (" + kind + ") error", ex);
                throw new OlapBuilderErrorException(OlapBuilderMongoDbErrorCode.JSON_ERROR, kind, "[field:" + fieldName + "] " + ex.getMessage());
            }
        }
    }

    @Nullable
    protected Bson getColumnBasedFilter(Comparable incrLoadMarker)
    {
        final Bson filter;
        Bson incrFilter = null;
        if (incrLoadMarker != null)
        {
            incrFilter = Filters.gt(table.getIncrementalLoadColumnName(), incrLoadMarker.toString());
        }
        Bson partFilter = null;
        if (partitionKey != null)
        {
            // there is not Query for the incremental load or partition, go for an easy equal filter on partition column
            partFilter = Filters.eq(table.getPartitioningColumnName(), partitionKey.asPartitionKeyObject());
        }
        if (partFilter != null && incrFilter != null)
        {
            filter = Filters.and(partFilter, incrFilter);
        }
        else
        {
            filter = incrFilter != null ? incrFilter : partFilter;
        }
        return filter;
    }

    public Object partitionKeyForReplacement()
    {
        return partitionKey.asPartitionKeyObject();
    }

}
